<?php
/**
 * Helper class for custom elementor widgets
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main Elementor Factory Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Brayn_Elementor_Factory {
	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Brayn_Elementor_Factory The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Brayn_Elementor_Factory An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {

		add_action( 'plugins_loaded', [ $this, 'init' ] );
		
		// On Editor - Register WooCommerce frontend hooks before the Editor init.
		// Priority = 5, in order to allow plugins remove/add their wc hooks on init.
		if ( ! empty( $_REQUEST['action'] ) && 'elementor' === $_REQUEST['action'] && is_admin() ) {
			add_action( 'init', [ $this, 'register_wc_hooks' ], 5 );
		}
	}

	/**
	 * Initialize the module
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Elementor not active.
		if ( ! did_action( 'elementor/loaded' ) ) {
			return;
		}

		// Elementor version checks.
		if ( defined( 'ELEMENTOR_VERSION' ) && ! version_compare( ELEMENTOR_VERSION, '2.7.4', '>=' ) ) {
			add_action( 'admin_notices', 'brayn_core_notice_elementor_version' );
			return;
		}

		add_filter( 'elementor/controls/animations/additional_animations', [ $this, 'add_custom_entrance_animation' ], 10, 1 );

		add_action( 'elementor/elements/categories_registered', [ $this, 'add_elementor_widget_categories' ] );
		add_action( 'elementor/widgets/widgets_registered', [ $this, 'init_widgets' ] );
		//add_action( 'elementor/controls/controls_registered', [ $this, 'init_controls' ] );
	}

	public function add_custom_entrance_animation( $animations ) {
		$animations = [
			'Brayn Animation'  => [
				'brayn_revealLeft'   => 'Reveal Left',
				'brayn_revealRight'  => 'Reveal Right',
				'brayn_revealTop'    => 'Reveal Top',
				'brayn_revealBottom' => 'Reveal Bottom',
			],
		];

		return $animations;
	}

	/**
	 * Register our new widget category to elementor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function add_elementor_widget_categories( $elements_manager ) {

		$elements_manager->add_category(
			'brayn-theme',
			[
				'title' => __( 'Brayn', 'brayn-core' ),
				'icon'  => 'fa fa-plug',
			]
		);
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-portfolio-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-scroll-sign-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-portfolio-slide-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-post-column-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-testimonial-slide-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-button-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-profile-card-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-circle-progress-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-carousel-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-metro-gallery-widget.php' );
		require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-masonry-gallery-widget.php' );

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Portfolio_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Portfolio_Slide_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Post_Column_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Carousel_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Metro_Gallery_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Masonry_Gallery_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Testimonial_Slide_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Profile_Card_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Button_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Scroll_Sign_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Circle_Progress_Widget() );
		
		/**
		 * WooCommerce widgets
		 *
		 * @since 2.0.0
		 */
		if ( brayn_core_woo_active() ) {
			require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-woocommerce-products.php' );
			require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-woocommerce-categories.php' );
			require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-woocommerce-onsale.php' );
			require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-woocommerce-featured.php' );
			require_once( BRAYN_CORE_PATH . '/inc/elementor/widgets/class-brayn-elementor-woocommerce-product-slide.php' );

			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Woocommerce_Products_Widget() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Woocommerce_Categories_Widget() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Woocommerce_Onsale_Widget() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Woocommerce_Featured_Widget() );
			\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Brayn_Elementor_Woocommerce_Product_Slide_Widget() );
		}

	}

	public function register_wc_hooks() {
		if ( brayn_core_woo_active() ) {
			wc()->frontend_includes();
		}
	}
}

Brayn_Elementor_Factory::instance();
