<?php
/**
 * Custom widgets included with this plugin
 *
 * @package Wip-Themes Core
 * @since 1.0
 */

/**
 * Load some helpers for custom widgets.
 *
 * @return void
 */
function brayn_core_admin_widget_enqueue_scripts( $hook ) {
	if ( 'widgets.php' !== $hook ) {
		return;
	}

	wp_enqueue_media();
	wp_enqueue_style( 'wip-themes-admin-widgets', esc_url( BRAYN_CORE_URL . '/assets/css/admin-widgets.css' ), array(), '1.0' );
	wp_enqueue_script( 'wip-themes-admin-widgets', esc_url( BRAYN_CORE_URL . '/assets/js/admin-widgets.js' ), array( 'jquery' ), '1.0', true );
}
add_action( 'admin_enqueue_scripts', 'brayn_core_admin_widget_enqueue_scripts' );

/**
 * Facebook like box Widget.
 */
class Wip_Themes_FB_Likebox_Widget extends WP_Widget {
	function __construct() {
		$widget_ops = array(
			'classname' => 'widget_99crv_fb_likebox',
			'description' => esc_html__( 'a Facebook Like Box.', 'brayn-core' ),
		);
		$control_ops = array(
			'width' => 400,
			'height' => 200,
		);
		parent::__construct( 'widget-99crv-fb-likebox', esc_html__( 'WIP-Themes: Facebook Like Box', 'brayn-core' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {
		extract( $args, EXTR_SKIP );

		$title 		= empty( $instance['title'] ) ? '' : $instance['title'];
		$page_url 	= empty( $instance['page_url'] ) ? '' : $instance['page_url'];
		$header 	= empty( $instance['header'] ) ? '' : $instance['header'];
		$faces 		= empty( $instance['faces'] ) ? '' : $instance['faces'];
		$tabs 		= empty( $instance['tabs'] ) ? 'false' : $instance['tabs'];

		echo $before_widget;

		if ( $title ) {
			echo $before_title . wp_kses_post( $title ) . $after_title;
		}

		if ( empty( $page_url ) ) {

			print '<em>' . esc_html__( 'No Facebook Page URL provided', 'brayn-core' ) . '</em>';

		} else {
			?>

<div class="_99crv-facebook-likebox-widget">

<div id="fb-root"></div>
<script>
<?php _99crv_core_fb_sdkjs(); ?>
(function(d, s, id) {
	var js, fjs = d.getElementsByTagName(s)[0];
	if (d.getElementById(id))
		return;
	js = d.createElement(s);
	js.id = id;
	js.src = "//connect.facebook.net/en_US/sdk.js";
	fjs.parentNode.insertBefore(js, fjs);
}(document, 'script', 'facebook-jssdk'));
</script>
<div class="fb-page" data-href="<?php echo esc_url( $page_url ); ?>" data-hide-cover="<?php echo ( empty( $header ) ? 'true' : 'false' ); ?>" data-show-facepile="<?php echo ( empty( $faces ) ? 'false' : 'true' ); ?>" data-tabs="<?php echo esc_attr( $tabs ); ?>" data-adapt-container-width="true" data-width="500"></div>

</div>

			<?php
		}

		echo $after_widget;

	}

	function update( $new_instance, $old_instance ) {
		$instance 				= $old_instance;
		$instance['title'] 		= strip_tags( $new_instance['title'] );
		$instance['page_url'] 	= strip_tags( $new_instance['page_url'] );
		$instance['header'] 	= strip_tags( $new_instance['header'] );
		$instance['faces'] 		= strip_tags( $new_instance['faces'] );
		$instance['tabs'] 		= strip_tags( $new_instance['tabs'] );

		return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args( (array) $instance, array(
			'title' 	=> '',
			'page_url' 	=> '',
			'header' 	=> 'yes',
			'faces' 	=> 'yes',
			'tabs' 		=> 'false',
			)
		);
		$title 		= strip_tags( $instance['title'] );
		$page_url 	= strip_tags( $instance['page_url'] );
		$header 	= strip_tags( $instance['header'] );
		$faces 		= strip_tags( $instance['faces'] );
		$tabs 		= strip_tags( $instance['tabs'] );
?>

	<p>
		<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php print esc_html__( 'Title:', 'brayn-core' ); ?>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
		</label>
	</p>
	<p>
		<label for="<?php echo esc_attr( $this->get_field_id( 'page_url' ) ); ?>"><?php print esc_html__( 'Facebook Page URL', 'brayn-core' ); ?>:
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'page_url' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'page_url' ) ); ?>" type="text" value="<?php echo esc_attr( $page_url ); ?>" />
		</label>
	</p>
	<p>
		<label for="<?php echo esc_attr( $this->get_field_id( 'tabs' ) ); ?>"><?php print esc_html__( 'Tabs to render', 'brayn-core' ); ?>:
			<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'tabs' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'tabs' ) ); ?>">
				<option value="false" <?php selected( 'false', $tabs, true ); ?>><?php print esc_html__( 'Hide tabs', 'brayn-core' ); ?></option>
				<option value="timeline" <?php selected( 'timeline', $tabs, true ); ?>><?php print esc_html__( 'Timeline', 'brayn-core' ); ?></option>
				<option value="events" <?php selected( 'events', $tabs, true ); ?>><?php print esc_html__( 'Events', 'brayn-core' ); ?></option>
				<option value="messages" <?php selected( 'messages', $tabs, true ); ?>><?php print esc_html__( 'Messages', 'brayn-core' ); ?></option>
				<option value="timeline,events,messages" <?php selected( 'timeline,events,messages', $tabs, true ); ?>><?php print esc_html__( 'Show all', 'brayn-core' ); ?></option>
			</select>
		</label>
	</p>
	<p>
		<label for="<?php echo esc_attr( $this->get_field_id( 'header' ) ); ?>">
			<input type="checkbox" value="yes" id="<?php echo esc_attr( $this->get_field_id( 'header' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'header' ) ); ?>" <?php checked( 'yes', $header, true ); ?> /> <?php print esc_html__( 'Show Header?', 'brayn-core' ); ?>
		</label>
	</p>
	<p>
		<label for="<?php echo esc_attr( $this->get_field_id( 'faces' ) ); ?>">
			<input type="checkbox" value="yes" id="<?php echo esc_attr( $this->get_field_id( 'faces' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'faces' ) ); ?>" <?php checked( 'yes', $faces, true ); ?> /> <?php print esc_html__( 'Show profile photos when friends like this?', 'brayn-core' ); ?>
		</label>
	</p>
			
<?php
	}
}

function _99crv_core_fb_sdkjs() {
	?>
	window.fbAsyncInit = function() {
		FB.init({appId: '169915343548301',autoLogAppEvents: true,xfbml: true, version: 'v2.9'});
		FB.AppEvents.logPageView();
		FB.Event.subscribe('xfbml.render', function(){
			jQuery(window).trigger('FBload');
		});
	};
	<?php
}

/**
 * About widget.
 *
 * @since version 1.0
 */
class Wip_Themes_About_Widget extends WP_Widget {

	function __construct() {
		$widget_ops = array(
		 'classname' => 'widget_99crv_about',
		 'description' => esc_html__( 'Show your short profile.', 'brayn-core' ),
		 );
		$control_ops = array(
			'width' => 400,
			'height' => 200,
		);
		WP_Widget::__construct( 'widget-99crv-about', esc_html__( 'WIP-Themes: About Me', 'brayn-core' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {
		extract( $args, EXTR_SKIP );

		$title = empty( $instance['title'] ) ? '' : $instance['title'];
		$profile_img = empty( $instance['profile_img'] ) ?  0 : intval( $instance['profile_img'] );
		$text = empty( $instance['text'] ) ?  '' : $instance['text'];
		$signature_img = empty( $instance['signature_img'] ) ? 0 : intval( $instance['signature_img'] );

		echo $before_widget;
		if ( '' !== $title ) {
			echo $before_title . esc_html( $title ) . $after_title;
		}

		print '<div class="_99crv-about-widget">' . "\n";

		if ( $profile_img ) {
			$profile_img_markup = wp_get_attachment_image( (int) $profile_img, 'full' );
			if ( $profile_img_markup ) {
				print '<div class="_99crv-about-widget-profile-image">' . $profile_img_markup . '</div>' . "\n";
			}
		}

		if ( '' !== $text ) {
			print wp_kses_post( wpautop( stripslashes( $text ) ) );
		}

		if ( $signature_img ) {
			$signature_img_markup = wp_get_attachment_image( (int) $signature_img, 'full' );
			if ( $signature_img_markup ) {
				print '<div class="about-wd-profile-signature">' . $signature_img_markup . '</div>' . "\n";
			}
		}

		print '</div>' . "\n";

		echo $after_widget;
	}

	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['profile_img'] = strip_tags( $new_instance['profile_img'] );
		$instance['text'] = stripslashes( $new_instance['text'] );
		$instance['signature_img'] = strip_tags( $new_instance['signature_img'] );

		return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance, array(
			'title' => '',
			'profile_img' => '',
			'text'  => '',
			'signature_img' => '',
			)
		);
		$title = strip_tags( $instance['title'] );
		$profile_img = strip_tags( $instance['profile_img'] );
		$text = stripslashes( $instance['text'] );
		$signature_img = strip_tags( $instance['signature_img'] );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php print esc_html__( 'Title:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</label>
		</p>
		<div class="ce-widget-image" id="wrp-<?php echo esc_attr( $this->get_field_id( 'profile_img' ) ); ?>">
			<?php print esc_html__( 'Profile image:', 'brayn-core' ); ?><br/>
			<em><small><?php print esc_html__( 'Please use a square image. Suggested dimension is 180px tall and wide.', 'brayn-core' ); ?></small></em><br/>
			<div class="ce-widget-image-preview">
			<?php
			if ( '' !== $profile_img ) {
				$profile_img_uri = wp_get_attachment_url( (int) $profile_img );
			?>
				<img src="<?php print esc_url( $profile_img_uri ); ?>" alt="" />
			<?php
			} else {
				print esc_html__( 'No image selected', 'brayn-core' );
			}
			?>
			</div>
			<a class="button ce-wd-ml" href="#" data-id="<?php echo esc_attr( $this->get_field_id( 'profile_img' ) ); ?>"><?php print esc_html__( 'Add image', 'brayn-core' ); ?></a>
			<a class="button ce-wd-ml-remove" href="#" data-id="<?php echo esc_attr( $this->get_field_id( 'profile_img' ) ); ?>"><?php print esc_html__( 'Remove image', 'brayn-core' ); ?></a>
			<input id="<?php echo esc_attr( $this->get_field_id( 'profile_img' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'profile_img' ) ); ?>" type="hidden" value="<?php echo esc_attr( $profile_img ); ?>" />
		</div>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'text' ) ); ?>"><?php print esc_html__( 'Profile text:', 'brayn-core' ); ?>
				<textarea class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'text' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'text' ) ); ?>" rows="10"><?php echo esc_textarea( $text ); ?></textarea>
			</label>
		</p>
		<div class="ce-widget-image" id="wrp-<?php echo esc_attr( $this->get_field_id( 'signature_img' ) ); ?>">
			<?php print esc_html__( 'Signature (optional):', 'brayn-core' ); ?><br/>
			<div class="ce-widget-image-preview">
			<?php
			if ( '' !== $signature_img ) {
				$signature_img_uri = wp_get_attachment_url( (int) $signature_img );
			?>
				<img src="<?php print esc_url( $signature_img_uri ); ?>" alt="" />
			<?php
			} else {
				print esc_html__( 'No image selected', 'brayn-core' );
			}
			?>
			</div>
			<a class="button ce-wd-ml" href="#" data-id="<?php echo esc_attr( $this->get_field_id( 'signature_img' ) ); ?>"><?php print esc_html__( 'Add image', 'brayn-core' ); ?></a>
			<a class="button ce-wd-ml-remove" href="#" data-id="<?php echo esc_attr( $this->get_field_id( 'signature_img' ) ); ?>"><?php print esc_html__( 'Remove image', 'brayn-core' ); ?></a>
			<input id="<?php echo esc_attr( $this->get_field_id( 'signature_img' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'signature_img' ) ); ?>" type="hidden" value="<?php echo esc_attr( $signature_img ); ?>" />
		</div>
		<?php
	}
}

/**
 * Custom widget for blog posts
 *
 * @since version 1.0
 */
class Wip_Themes_Blog_Posts_Widget extends WP_Widget {

	function __construct() {
		$widget_ops = array(
			'classname' => 'widget_99crv_posts',
			'description' => esc_html__( 'A list of your site\'s posts.', 'brayn-core' ),
			);
		$control_ops = array(
			'width' => 400,
			'height' => 200,
		);
		WP_Widget::__construct( 'widget-99crv-blog-post', esc_html__( 'WIP-Themes: Blog Posts', 'brayn-core' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {
		extract( $args, EXTR_SKIP );

		$title = empty( $instance['title'] ) ? esc_html__( 'Posts', 'brayn-core' ) : $instance['title'];
		$entry_numb = empty( $instance['entry_numb'] ) ?  6 : $instance['entry_numb'];
		$cat_id = empty( $instance['cat_id'] ) ?  0 : $instance['cat_id'];
		$orderby = empty( $instance['orderby'] ) ? 'date' : $instance['orderby'];
		$order = empty( $instance['order'] ) ? 'DESC' : $instance['order'];

		echo $before_widget;
		echo $before_title . wp_kses_post( $title ) . $after_title;

		$post_args = array(
			'post_type' => 'post',
			'post_status' => 'publish',
			'posts_per_page' => intval( $entry_numb ),
			'ignore_sticky_posts' => 1,
			'orderby' => $orderby,
			'order' => $order,
			'paged' => 1,
		);

		if ( '' !== $cat_id && intval( $cat_id ) > 0 ) {
			$post_args['cat'] = intval( $cat_id );
		}

		$p_query = new WP_Query( $post_args );

		if ( $p_query->have_posts() ) :

			print '<ul class="_99crv-widget-posts">' . "\n";

			while ( $p_query->have_posts() ) : $p_query->the_post();

				$thumbnail = '<span class="_99crv-widget-post-thumbnail"><img src="' . BRAYN_CORE_URL . '/assets/images/square-no-image.jpg" alt="no image" /></span>';

				if ( '' !== get_the_post_thumbnail( get_the_ID() ) ) {
					$thumbnail = '<span class="_99crv-widget-post-thumbnail">' . get_the_post_thumbnail( get_the_ID(), 'thumbnail' ) . '</span>';
				}

				$time_string = '<time class="_99crv-widget-entry-date published" datetime="%1$s">%2$s</time>';
				$time_string = sprintf(
					$time_string,
					get_the_date( DATE_W3C ),
					get_the_date()
				);

				print '<li>
				<a href="' . esc_url( get_permalink() ) . '" rel="bookmark" title="' . esc_attr( get_the_title() ) . '">
				' . wp_kses_post( $thumbnail ) . '
				<span class="_99crv-widget-post-title"><span>' . wp_kses_post( get_the_title() ) . '</span></span>
				' . wp_kses( $time_string, array(
					'time' => array(
						'class' => array(),
						'datetime' => array(),
					),
				) ) . '
				</a>
				</li>' . "\n";

			endwhile;

			wp_reset_postdata();

			print '</ul>';

		else :

			print '<div class="_99crv-widget-post-blank">' . esc_html__( 'No posts found!', 'brayn-core' ) . '</div>';

		endif;

		echo $after_widget;
	}

	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['entry_numb'] = strip_tags( $new_instance['entry_numb'] );
		$instance['cat_id'] = strip_tags( $new_instance['cat_id'] );
		$instance['orderby'] = strip_tags( $new_instance['orderby'] );
		$instance['order'] = strip_tags( $new_instance['order'] );

		return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance, array(
			'title'		=> '',
			'entry_numb' => 6,
			'cat_id'	=> '0',
			'orderby'	=> 'date',
			'order'		=> 'DESC',
			)
		);
		$title = strip_tags( $instance['title'] );
		$entry_numb = strip_tags( $instance['entry_numb'] );
		$cat_id = strip_tags( $instance['cat_id'] );
		$orderby = strip_tags( $instance['orderby'] );
		$order = strip_tags( $instance['order'] );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php print esc_html__( 'Title:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'entry_numb' ) ); ?>"><?php print esc_html__( 'Number of posts to show:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'entry_numb' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'entry_numb' ) ); ?>" type="text" value="<?php echo esc_attr( $entry_numb ); ?>" />
			</label>
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'cat_id' ) ); ?>"><?php print esc_html__( 'Category:', 'brayn-core' ); ?>
			<?php
				echo wp_kses( brayn_core_get_tax_lists( 'category', $this->get_field_name( 'cat_id' ), $this->get_field_id( 'cat_id' ), 'widefat', esc_html__( 'All Categories', 'brayn-core' ), $cat_id ), array(
					'select' => array(
						'id' => array(),
						'name' => array(),
						'class' => array(),
					),
					'option' => array(
						'value' => array(),
						'selected' => array(),
					),
				) );
			?>
			</label>
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>"><?php print esc_html__( 'Order By:', 'brayn-core' ); ?>
				<select class="widefat" name="<?php echo esc_attr( $this->get_field_name( 'orderby' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>">    
				<?php
					$ordby = array(
						'ID' => esc_html__( 'Post ID', 'brayn-core' ),
						'date' => esc_html__( 'Post Date', 'brayn-core' ),
						'title' => esc_html__( 'Post Title', 'brayn-core' ),
						'comment_count' => esc_html__( 'Comment Count', 'brayn-core' ),
						'rand' => esc_html__( 'Random Order', 'brayn-core' ),
					);

					foreach ( $ordby as $o => $name ) {
						echo '<option value="' . esc_attr( $o ) . '" ' . selected( $orderby, $o, false ) . '>' . esc_html( $name ) . '</option>' . "\n";
					}
					unset( $o );
				?>
				</select>
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>"><?php print esc_html__( 'Order:', 'brayn-core' ); ?>
				<select class="widefat" name="<?php echo esc_attr( $this->get_field_name( 'order' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>">    
					<?php
						$ordersel = array(
							'DESC' => esc_html__( 'Descending', 'brayn-core' ),
							'ASC' => esc_html__( 'Ascending', 'brayn-core' ),
						);
						foreach ( $ordersel as $or => $orname ) {
							echo '<option value="' . esc_attr( $or ) . '" ' . selected( $order, $or, false ) . '>' . esc_html( $orname ) . '</option>' . "\n";
						}
						unset( $or );
						unset( $orname );
					?>
				</select>
			</label>
		</p>

		<?php
	}
}

/**
 * Featured posts widget.
 *
 * @since version 1.0
 */
class Wip_Themes_Featured_Posts_Widget extends WP_Widget {
	function __construct() {
		$widget_ops = array(
			'classname' => 'widget_99crv_posts',
			'description' => esc_html__( 'A list of your featured posts.', 'brayn-core' ),
		);
		$control_ops = array(
			'width' => 400,
			'height' => 200,
		);
		WP_Widget::__construct( 'widget-99crv-featured-post', esc_html__( 'WIP-Themes: Featured Posts', 'brayn-core' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {
		extract( $args, EXTR_SKIP );

		$title = empty( $instance['title'] ) ? esc_html__( 'Featured Posts', 'brayn-core' ) : $instance['title'];
		$entry_numb = empty( $instance['entry_numb'] ) ?  6 : $instance['entry_numb'];
		$orderby = empty( $instance['orderby'] ) ? 'date' : $instance['orderby'];
		$order = empty( $instance['order'] ) ? 'DESC' : $instance['order'];
		
		$post_args = array(
			'post_type' 	=> 'post',
			'post_status' 	=> 'publish',
			'posts_per_page' => intval( $entry_numb ),
			'ignore_sticky_posts' => 1,
			'orderby' 		=> $orderby,
			'order' 		=> $order,
			'paged' 		=> 1,
			'meta_query' 			=> array(
				array(
					'key'     => '_wip_themes_featured',
					'value'   => array( '1' ),
					'compare' => 'IN',
				),
			)
		);

		$p_query = new WP_Query( $post_args );

		// no featured posts? leave it.
		if ( ! $p_query->have_posts() ) {
			return false;
		}

		echo $before_widget;
		echo $before_title . wp_kses_post( $title ) . $after_title;

		print '<ul class="_99crv-widget-posts">' . "\n";

			while ( $p_query->have_posts() ) : $p_query->the_post();

				$thumbnail = '<span class="_99crv-widget-post-thumbnail"><img src="' . BRAYN_CORE_URL . '/assets/images/square-no-image.jpg" alt="no image" /></span>';

				if ( '' !== get_the_post_thumbnail( get_the_ID() ) ) {
					$thumbnail = '<span class="_99crv-widget-post-thumbnail">' . get_the_post_thumbnail( get_the_ID(), 'thumbnail' ) . '</span>';
				}

				$time_string = '<time class="_99crv-widget-entry-date published" datetime="%1$s">%2$s</time>';
				$time_string = sprintf(
					$time_string,
					get_the_date( DATE_W3C ),
					get_the_date()
				);

				print '<li>
				<a href="' . esc_url( get_permalink() ) . '" rel="bookmark" title="' . esc_attr( get_the_title() ) . '">
				' . wp_kses_post( $thumbnail ) . '
				<span class="_99crv-widget-post-title"><span>' . wp_kses_post( get_the_title() ) . '</span></span>
				' . wp_kses( $time_string, array(
					'time' => array(
						'class' => array(),
						'datetime' => array(),
					),
				) ) . '
				</a>
				</li>' . "\n";

			endwhile;
			wp_reset_postdata();

		print '</ul>';

		echo $after_widget;
	}

	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['entry_numb'] = strip_tags( $new_instance['entry_numb'] );
		$instance['orderby'] = strip_tags( $new_instance['orderby'] );
		$instance['order'] = strip_tags( $new_instance['order'] );

		return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance, array(
			'title'         => '',
			'entry_numb'     => 6,
			'orderby'        => 'date',
			'order'            => 'DESC',
			)
		);
		$title = strip_tags( $instance['title'] );
		$entry_numb = strip_tags( $instance['entry_numb'] );
		$orderby = strip_tags( $instance['orderby'] );
		$order = strip_tags( $instance['order'] );

		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php print esc_html__( 'Title:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</label>
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'entry_numb' ) ); ?>"><?php print esc_html__( 'Number of posts to show:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'entry_numb' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'entry_numb' ) ); ?>" type="text" value="<?php echo esc_attr( $entry_numb ); ?>" />
			</label>
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>"><?php print esc_html__( 'Order By:', 'brayn-core' ); ?>
				<select class="widefat" name="<?php echo esc_attr( $this->get_field_name( 'orderby' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'orderby' ) ); ?>">    
				<?php
					$ordby = array(
						'ID' 	=> esc_html__( 'Post ID', 'brayn-core' ),
						'date' 	=> esc_html__( 'Post Date', 'brayn-core' ),
						'title' => esc_html__( 'Post Title', 'brayn-core' ),
						'comment_count' => esc_html__( 'Comment Count', 'brayn-core' ),
						'rand' 	=> esc_html__( 'Random Order', 'brayn-core' ),
					);
					foreach ( $ordby as $o => $name ) {
						echo '<option value="' . esc_attr( $o ) . '" ' . selected( $orderby, $o, false ) . '>' . esc_html( $name ) . '</option>' . "\n";
					}
					unset( $o );
				?>
				</select>
			</label>
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>"><?php print esc_html__( 'Order:', 'brayn-core' ); ?>
				<select class="widefat" name="<?php echo esc_attr( $this->get_field_name( 'order' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>">    
				<?php
					$ordersel = array(
						'DESC' => esc_html__( 'Descending', 'brayn-core' ),
						'ASC' => esc_html__( 'Ascending', 'brayn-core' ),
					);
					foreach ( $ordersel as $or => $orname ) {
						echo '<option value="' . esc_attr( $or ) . '" ' . selected( $order, $or, false ) . '>' . esc_html( $orname ) . '</option>' . "\n";
					}
					unset( $or );
				?>
				</select>
			</label>
		</p>

		<?php
	}
}

/**
 * Instagram widget.
 *
 * @since version 1.0
 */
class Wip_Themes_Instagram_Widget extends WP_Widget {

	function __construct() {
		$widget_ops = array(
			'classname' => 'widget_99crv_instagram',
			'description' => esc_html__( 'Show latest Instagram thumbnails.', 'brayn-core' ),
		);
		$control_ops = array(
			'width' => 400,
			'height' => 200,
		);
		WP_Widget::__construct( 'widget-99crv-instagram', esc_html__( 'WIP-Themes: Instagram', 'brayn-core' ), $widget_ops, $control_ops );
	}

	function widget( $args, $instance ) {
		extract( $args, EXTR_SKIP );

		$title = empty( $instance['title'] ) ? '' : $instance['title'];
		$limit = empty( $instance['limit'] ) ?  20 : intval( $instance['limit'] );
		$username = empty( $instance['username'] ) ?  '' : esc_html( $instance['username'] );
		$link_text = empty( $instance['link'] ) ? '' : $instance['link'];

		if ( '' === $username ) {
			return false;
		}

		echo $before_widget;
		if ( '' !== $title ) {
			echo $before_title . esc_html( $title ) . $after_title;
		}

		print '<div class="_99crv-ig-wrap">' . "\n";

		wip_themes_core_print_instagram( $username, '', 'ig-item', $limit, 'all', 'thumbnail' );

		if ( '' !== $link_text ) {
			print '<p><a class="_99crv-ig-link-account" href="' . esc_url( trailingslashit( '//instagram.com/' . esc_attr( trim( str_replace( '@', '', $username ) ) ) ) ) . '" target="_blank" rel="nofollow">' . esc_html( $link_text ) . '</a></p>';
		}

		print '</div>' . "\n";

		echo $after_widget;
	}

	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['limit'] = intval( $new_instance['limit'] );
		$instance['username'] = strip_tags( $new_instance['username'] );
		$instance['link'] = esc_html( $new_instance['link'] );

		return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance, array(
			'title' => '',
			'limit' => 9,
			'username'  => '',
			'link' => '',
			)
		);
		$title = strip_tags( $instance['title'] );
		$limit = intval( $instance['limit'] );
		$username = strip_tags( $instance['username'] );
		$link = strip_tags( $instance['link'] );

		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php print esc_html__( 'Title:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>"><?php print esc_html__( 'Instagram username:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'username' ) ); ?>" type="text" value="<?php echo esc_attr( $username ); ?>" />
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>"><?php print esc_html__( 'Number of image to show:', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'limit' ) ); ?>" type="number" value="<?php echo esc_attr( $limit ); ?>" />
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>"><?php print esc_html__( 'Link text (optional):', 'brayn-core' ); ?>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'link' ) ); ?>" type="text" value="<?php echo esc_attr( $link ); ?>" />
			</label>
			<span class="description"><?php esc_html_e( 'Show a link to your Instagram account? leave blank this field to hide the link.', 'brayn-core' ); ?></span>
		</p>
		<?php
	}
}
