<?php
/**
 * custom elementor image carousel widget
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 1.0
 */

/**
 * Elementor Image Carousel Widget.
 *
 * @since 1.0.0
 */
class Brayn_Elementor_Carousel_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-carousel';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Image Slide', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-slides';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'images',
			[
				'label' => __( 'Add Images', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::GALLERY,
				'default' => [],
				'show_label' => false,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
				'default' => 'brayn-post-cover',
				'exclude' => [ 'custom', 'thumbnail', 'medium', 'medium_large', 'brayn-post-square-small', 'brayn-post-potrait' ],
				'separator' => 'none',
			]
		);

		$this->add_control(
			'gallery_link',
			[
				'label' => __( 'Link', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'file',
				'options' => [
					'file' => __( 'Media File', 'brayn-core' ),
					'none' => __( 'None', 'brayn-core' ),
				],
			]
		);

		$this->add_control(
			'open_lightbox',
			[
				'label' => __( 'Lightbox', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => __( 'Default', 'brayn-core' ),
					'yes' => __( 'Yes', 'brayn-core' ),
					'no' => __( 'No', 'brayn-core' ),
				],
				'condition' => [
					'gallery_link' => 'file',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings    = $this->get_settings_for_display();
		
		if ( empty( $settings['images'] ) ) {
			return;
		}

		$slides = [];

		foreach ( $settings['images'] as $index => $attachment ) {
			$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src( $attachment['id'], 'thumbnail', $settings );
			$image_html = '<img class="swiper-slide-image" src="' . esc_attr( $image_url ) . '" alt="' . esc_attr( \Elementor\Control_Media::get_image_alt( $attachment ) ) . '" />';
		
			$link_tag = '';

			$link = $this->get_link_url( $attachment, $settings );
			if ( $link ) {
				$link_key = 'link_' . $index;

				$this->add_render_attribute( $link_key, [
					'href' => $link['url'],
					'data-elementor-open-lightbox' => $settings['open_lightbox'],
					'data-elementor-lightbox-slideshow' => $this->get_id(),
					'data-elementor-lightbox-index' => $index,
				] );

				if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
					$this->add_render_attribute( $link_key, [
						'class' => 'elementor-clickable',
					] );
				}

				$link_tag = '<a ' . $this->get_render_attribute_string( $link_key ) . '>';
			}

			$slide_html = '<div class="swiper-slide">';
			$slide_html .= '<div class="br-carousel-item">';
			$slide_html .= '<div class="br-carousel-bg"></div>';

			$slide_html .= $link_tag . $image_html;

			if ( $link ) {
				$slide_html .= '</a>';
			}

			$slide_html .= '</div>';
			$slide_html .= '</div>';

			$slides[] = $slide_html;
		}

		if ( empty( $slides ) ) {
			return;
		}

		?>
		<div class="br-carousel-gallery br-gallery">
			<div class="br-carousel-gallery-container swiper-container">
				<div class="swiper-wrapper">
					<?php echo implode( '', $slides ); ?>
				</div>

				<!-- Prev / Next button -->
				<div class="br-gallery-carousel-buttons">
					<div class="br-gallery-carousel-btn-prev">
						<svg class="svg-icon" viewBox="0 0 20 20">
							<path d="M18.271,9.212H3.615l4.184-4.184c0.306-0.306,0.306-0.801,0-1.107c-0.306-0.306-0.801-0.306-1.107,0
							L1.21,9.403C1.194,9.417,1.174,9.421,1.158,9.437c-0.181,0.181-0.242,0.425-0.209,0.66c0.005,0.038,0.012,0.071,0.022,0.109
							c0.028,0.098,0.075,0.188,0.142,0.271c0.021,0.026,0.021,0.061,0.045,0.085c0.015,0.016,0.034,0.02,0.05,0.033l5.484,5.483
							c0.306,0.307,0.801,0.307,1.107,0c0.306-0.305,0.306-0.801,0-1.105l-4.184-4.185h14.656c0.436,0,0.788-0.353,0.788-0.788
							S18.707,9.212,18.271,9.212z"></path>
						</svg>
					</div>
					<div class="br-gallery-carousel-btn-next">
						<svg class="svg-icon" viewBox="0 0 20 20">
							<path d="M1.729,9.212h14.656l-4.184-4.184c-0.307-0.306-0.307-0.801,0-1.107c0.305-0.306,0.801-0.306,1.106,0
							l5.481,5.482c0.018,0.014,0.037,0.019,0.053,0.034c0.181,0.181,0.242,0.425,0.209,0.66c-0.004,0.038-0.012,0.071-0.021,0.109
							c-0.028,0.098-0.075,0.188-0.143,0.271c-0.021,0.026-0.021,0.061-0.045,0.085c-0.015,0.016-0.034,0.02-0.051,0.033l-5.483,5.483
							c-0.306,0.307-0.802,0.307-1.106,0c-0.307-0.305-0.307-0.801,0-1.105l4.184-4.185H1.729c-0.436,0-0.788-0.353-0.788-0.788
							S1.293,9.212,1.729,9.212z"></path>
						</svg>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Retrieve image carousel link URL.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param array $attachment
	 * @param object $instance
	 *
	 * @return array|string|false An array/string containing the attachment URL, or false if no link.
	 */
	private function get_link_url( $attachment, $instance ) {
		if ( 'none' === $instance['gallery_link'] ) {
			return false;
		}

		return [
			'url' => wp_get_attachment_url( $attachment['id'] ),
		];
	}
}
