<?php
/**
 * Custom elementor profile card widget
 *
 * @package    Elementor
 * @subpackage Brayn_Core
 * @since      1.0
 */

/**
 * Elementor Profile card Widget.
 *
 * @since 1.0.0
 */
class Brayn_Elementor_Profile_Card_Widget extends \Elementor\Widget_Base
{

	/**
	 * Get widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-profilecard';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Profile Card', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-person';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'profile_img',
			[
				'label'       => __( 'Profile Image', 'brayn-core' ),
				'type'        => \Elementor\Controls_Manager::MEDIA,
				'default'     => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name'    => 'thumbnail',
				'exclude' => [],
				'include' => [],
				'default' => 'brayn-post-potrait',
			]
		);

		$this->add_control(
			'profile_name',
			[
				'label' => esc_html__( 'Name', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'input_type' => 'text',
				'placeholder' => 'John Due',
				'default' => '',
				'description' => esc_html__( 'Enter the person name', 'brayn-core' ),
			]
		);

		$this->add_control(
			'profile_pos',
			[
				'label' => esc_html__( 'Business Role', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'input_type' => 'text',
				'placeholder' => '',
				'default' => '',
				'description' => esc_html__( 'Enter the business role/position, e.g "CEO", "Lead Marketing", etc.', 'brayn-core' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'link_section',
			[
				'label' => esc_html__( 'Links', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'link_name',
			[
				'label'       => __( 'Link text', 'brayn-core' ),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'default'     => '',
				'label_block' => true,
				'description' => esc_html__( 'Shortname of social media (e.g Tw for Twitter, Fb for Facebook), or simply put "More info" and linked to another page', 'brayn-core' ),
			]
		);

		$repeater->add_control(
			'link_url',
			[
				'label'   => __( 'Link', 'brayn-core' ),
				'type'    => \Elementor\Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'https://your-link.com', 'brayn-core' ),
				'default' => [
					'url' => '#',
				],
			]
		);

		$this->add_control(
			'links',
			[
				'label'  => __( 'Links', 'brayn-core' ),
				'type'   => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'link_name' => esc_html__( 'Fb', 'brayn-core' ),
						'link_url'  => [
							'url'         => 'https://www.facebook.com',
							'is_external' => false,
						]
					],
				],
				'title_field' => '{{{ link_name }}}',
				'show_label' => false,
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		print '<div class="team-card">' . "\n";

			print '<div class="team-avatar">' . "\n";
			print \Elementor\Group_Control_Image_Size::get_attachment_image_html( $settings, 'thumbnail', 'profile_img' );
			print '</div>' . "\n";

			print '<div class="team-info">' . "\n";
			print '<h4>' . ( isset( $settings['profile_name'] ) ? $settings['profile_name'] : '' ) . '</h4>' . "\n";
			if ( isset( $settings['profile_pos'] ) && ! empty( $settings['profile_pos'] ) ) {
				print '<span class="role">' . $settings['profile_pos'] . '</span>' . "\n";
			}

			if ( $settings['links'] ) {
				print '<ul class="team-social">' . "\n";

				$a = 0;
				foreach( $settings['links'] as $item ) {
					$a++;
					print '<li>';

					if ( ! empty( $item['link_url']['url'] ) ) {
						$this->add_render_attribute( 'button_'.$a, 'href', $item['link_url']['url'] );

						if ( $item['link_url']['is_external'] ) {
							$this->add_render_attribute( 'button_'.$a, 'target', '_blank' );
						}

						if ( $item['link_url']['nofollow'] ) {
							$this->add_render_attribute( 'button_'.$a, 'rel', 'nofollow' );
						}
					}

					$this->add_render_attribute( 'button_'.$a, 'class', 'team-social-link' );

					print '<a ' . $this->get_render_attribute_string( 'button_'.$a ) . '>' . stripslashes( $item['link_name'] ) . '</a>';

					print '</li>';
				}
				unset( $a );

				print '</ul>' . "\n";
			}

			print '</div>' . "\n";
		
		print '</div>' . "\n";

	}
}
