<?php
/**
 * custom elementor woocommerce products slide
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 2.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Color;

class Brayn_Elementor_Woocommerce_Product_Slide_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-woo-slide';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Woo: Products Carousel', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-carousel';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 2.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'query_type',
			[
				'label' => esc_html__( 'Product Type', 'brayn-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'date',
				'description' => '',
				'options' => [
					'date'     => esc_html__( 'Latest Products', 'brayn-core' ),
					'rating'   => esc_html__( 'Top Rated Products', 'brayn-core' ),
					'sale'     => esc_html__( 'Sale Products', 'brayn-core' ),
					'featured' => esc_html__( 'Featured Products', 'brayn-core' ),
					'sales'    => esc_html__( 'Best Selling Products', 'brayn-core' ),
				],
			]
		);

		$this->add_control(
			'number_post',
			[
				'label' => esc_html__( 'Number of products', 'brayn-core' ),
				'type' => Controls_Manager::TEXT,
				'input_type' => 'number',
				'placeholder' => '',
				'default' => 4,
				'description' => '',
			]
		);

		$this->add_responsive_control(
			'slide_col',
			[
				'label' => esc_html__( 'Columns', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'col' => [
						'min' => 1,
						'max' => 9,
						'step' => 1,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 4,
					'unit' => 'col',
				],
				'tablet_default' => [
					'size' => 3,
					'unit' => 'col',
				],
				'mobile_default' => [
					'size' => 2,
					'unit' => 'col',
				],
			]
		);

		$cats = brayn_core_taxonomies_options( 'product_cat', true );
		$this->add_control(
			'category',
			[
				'label' => esc_html__( 'Product category', 'brayn-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'all',
				'description' => '',
				'options' => $cats,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label' => esc_html__( 'Order by', 'brayn-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'date',
				'description' => '',
				'options' => [
					'date'       => esc_html__( 'Published date', 'brayn-core' ),
					'menu_order' => esc_html__( 'The menu order', 'brayn-core' ),
					'popularity' => esc_html__( 'Number of sales', 'brayn-core' ),
					'rating'     => esc_html__( 'Average rating', 'brayn-core' ),
					'title'      => esc_html__( 'Product title', 'brayn-core' ),
					'rand'       => esc_html__( 'Random', 'brayn-core' ),
				],
				'condition' => [
					'query_type' => ['featured', 'sale'],
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label' => __( 'Order', 'brayn-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc' => __( 'ASC', 'brayn-core' ),
					'desc' => __( 'DESC', 'brayn-core' ),
				],
				'condition' => [
					'query_type' => ['featured', 'sale'],
				],
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label' => __( 'Autoplay', 'brayn-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => '',
				'label_on' => 'On',
				'label_off' => 'Off',
			]
		);

		$this->add_control(
			'autoplay_delay',
			[
				'label' => __( 'Autoplay Delay', 'brayn-core' ),
				'description' => __( 'Delay between transitions (in ms).', 'brayn-core' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 3000,
				'min' => 0,
				'max' => 10000,
				'condition' => [
					'autoplay' => 'yes',
				],
			]
		);

		$this->add_control(
			'speed',
			[
				'label' => __( 'Slide Speed', 'brayn-core' ),
				'description' => __( 'Duration of transition between slides (in ms).', 'brayn-core' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 300,
				'min' => 100,
				'max' => 10000,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'slide_nav_section',
			[
				'label' => esc_html__( 'Navigation', 'brayn-core' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pagination',
			[
				'label' => __( 'Bullet Pagination', 'brayn-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => 'On',
				'label_off' => 'Off',
			]
		);

		$this->add_control(
			'pag_size',
			[
				'label' => esc_html__( 'Size', 'brayn-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' =>['px'],
				'range' => [
					'px' => [
						'min' => 6,
						'max' => 30,
						'step' => 1,
					],
				],
				'default' =>[
					'unit' => 'px',
					'size' => 6,
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'pag_spacing',
			[
				'label' => esc_html__( 'Spacing', 'brayn-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' =>['px'],
				'range' => [
					'px' => [
						'min' => 2,
						'max' => 30,
						'step' => 1,
					],
				],
				'default' =>[
					'unit' => 'px',
					'size' => 6,
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-container-horizontal>.swiper-pagination-bullets .swiper-pagination-bullet' => 'margin: 0px {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'pag_bullet_border',
				'selector' => '{{WRAPPER}} .swiper-pagination-bullet',
			]
		);

		$this->start_controls_tabs( 'pag_button_style' );

		$this->start_controls_tab(
			'pag_button_normal',
			[
				'label' => __( 'Normal', 'brayn-core' ),
			]
		);

		$this->add_control(
			'pag_color',
			[
				'label'     => __( 'Color', 'brayn-core' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_1,
				],
				'default'   => '#dadada',
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet' => 'background: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'pag_button_active',
			[
				'label' => __( 'Active', 'brayn-core' ),
			]
		);

		$this->add_control(
			'pag_color_active',
			[
				'label'     => __( 'Color', 'brayn-core' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => Scheme_Color::get_type(),
					'value' => Scheme_Color::COLOR_1,
				],
				'default'   => '#46ad2a',
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'background: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	private function get_shortcode() {
		$settings = $this->get_settings();

		$attributes = [
			'limit'    => $settings['number_post'],
			'columns'  => $settings['slide_col']['size'],
			'paginate' => false,
			'class'    => 'e-brayn-product-slide swiper-container',
		];


		$type = $settings['query_type'];

		switch ( $type ) {
			case 'rating':
				$attributes['top_rated'] = true;
				$attributes['orderby'] = 'rating';
				$attributes['order'] = 'desc';
				break;
			case 'sale':
				$attributes['on_sale'] = true;
				$attributes['orderby'] = $settings['orderby'];
				$attributes['order']   = $settings['order'];
				break;
			case 'featured':
				$attributes['visibility'] = 'featured';
				$attributes['orderby'] = $settings['orderby'];
				$attributes['order']   = $settings['order'];
				break;
			case 'sales':
				$attributes['best_selling'] = true;
				break;
			default:
				$attributes['orderby'] = 'date';
				$attributes['order'] = 'desc';
		}

		if ( 'all' !== $settings['category'] ) {
			$attributes['category'] = $settings['category'];
		}
		
		$this->add_render_attribute( 'shortcode', $attributes );

		$shortcode = sprintf( '[products %s]', $this->get_render_attribute_string( 'shortcode' ) );

		return $shortcode;
	}

	public function add_swiper_class( $classes ){
		$classes[] = 'swiper-slide';

		return $classes;
	}

	public function loop_start_inject() {
		$settings = $this->get_settings();

		$html = '<ul class="products swiper-wrapper columns-'. absint( $settings['slide_col']['size'] ) .'">';

		return $html;
	}

	public function loop_end_inject() {
		$settings = $this->get_settings();
		$html = '</ul>';
		if ( 'yes' === $settings['pagination'] ) {
			$html .= '<div class="swiper-pagination"></div>';
		}

		return $html;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 2.0.0
	 * @access protected
	 */
	public function render() {
		if ( ! brayn_core_woo_active() ) {
			return false;
		}

		$settings = $this->get_settings();
		$this->add_render_attribute(
			'product_swipe_wrap',
			[
				'class' => [ 'e-brayn-product-slide-container'  ],
				'data-colnumb' => '{"def":"'.$settings['slide_col']['size'].'","tablet":"'.$settings['slide_col_tablet']['size'].'","mobile":"'.$settings['slide_col_mobile']['size'].'"}',
				'data-autoplay' => '{"enabled":"'.$settings['autoplay'].'","delay":"'.$settings['autoplay_delay'].'"}',
				'data-speed' => $settings['speed'],
			]
		);

		print '<div ' . $this->get_render_attribute_string( 'product_swipe_wrap' ) . '>'  . "\n";

		add_filter( 'woocommerce_product_loop_start', [$this, 'loop_start_inject'], 999 );
		add_filter( 'woocommerce_product_loop_end', [$this, 'loop_end_inject'], 999 );
		add_filter( 'woocommerce_post_class', [$this, 'add_swiper_class'], 999 );
		echo do_shortcode( $this->get_shortcode() );
		remove_filter( 'woocommerce_post_class', [$this, 'add_swiper_class'], 999 );
		remove_filter( 'woocommerce_product_loop_start', [$this, 'loop_start_inject'], 999 );
		remove_filter( 'woocommerce_product_loop_end', [$this, 'loop_end_inject'], 999 );

		print '</div>';
	}

	public function render_plain_content() {
		echo $this->get_shortcode();
	}
}
