<?php
/**
 * custom elementor woocommerce product widget
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 2.0.0
 */

class Brayn_Elementor_Woocommerce_Products_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-woo-products';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Woo: Products', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-products';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 2.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'number_post',
			[
				'label' => esc_html__( 'Number of products', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'input_type' => 'number',
				'placeholder' => '',
				'default' => 4,
				'description' => '',
			]
		);

		$this->add_control(
			'post_col',
			[
				'label' => esc_html__( 'Columns', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' =>['col'],
				'range' => [
					'col' => [
						'min' => 2,
						'max' => 6,
						'step' => 1,
					],
				],
				'default' =>[
					'unit' => 'col',
					'size' => 4,
				],
			]
		);

		$cats = brayn_core_taxonomies_options( 'product_cat' );
		$this->add_control(
			'category',
			[
				'label' => esc_html__( 'Product category', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'all',
				'description' => '',
				'options' => $cats,
			]
		);

		$this->add_control(
			'sort_by',
			[
				'label' => esc_html__( 'Sort by', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'date',
				'description' => '',
				'options' => [
					'date'       => esc_html__( 'Latest', 'brayn-core' ),
					'sales'      => esc_html__( 'Sales', 'brayn-core' ),
					'rating'     => esc_html__( 'Average Rating', 'brayn-core' ),
					'low_price'  => esc_html__( 'Price: low to high', 'brayn-core' ),
					'high_price' => esc_html__( 'Price: high to low', 'brayn-core' ),
					'rand'       => esc_html__( 'Random', 'brayn-core' ),
				],
			]
		);

		$this->end_controls_section();
	}

	public function loop_columns() {
		$settings    = $this->get_settings_for_display();
		$column      = $settings['post_col']['size'];
		return absint( $column );
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 2.0.0
	 * @access protected
	 */
	protected function render() {
		$settings    = $this->get_settings_for_display();
		$post_number = $settings['number_post'];
		$column      = $settings['post_col']['size'];
		$sort_by     = $settings['sort_by'];
		$category    = $settings['category'];

		if ( ! brayn_core_woo_active() ) {
			return false;
		}

		$product_visibility_term_ids = wc_get_product_visibility_term_ids();

		$order = ( $sort_by === 'low_price' ) ? "ASC" : "DESC";

		$query_args = array(
			'posts_per_page' => $post_number,
			'post_status'    => 'publish',
			'post_type'      => 'product',
			'no_found_rows'  => 1,
			'order'          => $order,
			'meta_query'     => array(),
			'tax_query'      => array(
				'relation' => 'AND',
			),
		); // WPCS: slow query ok.

		$query_args['tax_query'][] = array(
			'taxonomy' => 'product_visibility',
			'field'    => 'term_taxonomy_id',
			'terms'    => is_search() ? $product_visibility_term_ids['exclude-from-search'] : $product_visibility_term_ids['exclude-from-catalog'],
			'operator' => 'NOT IN',
		);
		$query_args['post_parent'] = 0;

		// eliminate out of stock items.
		$query_args['tax_query'][] = array(
			array(
				'taxonomy' => 'product_visibility',
				'field'    => 'term_taxonomy_id',
				'terms'    => $product_visibility_term_ids['outofstock'],
				'operator' => 'NOT IN',
			),
		); // WPCS: slow query ok.

		if ( $category !== 'all' && intval( $category ) > 0 ) {
			$query_args['tax_query'][] = array(
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'term_taxonomy_id',
					'terms'    => intval( $category ),
					'operator' => 'IN'
				),
			);
		}

		switch ( $sort_by ) {
			case 'low_price':
			case 'high_price':
				$query_args['meta_key'] = '_price'; // WPCS: slow query ok.
				$query_args['orderby']  = 'meta_value_num';
				break;
			case 'rand':
				$query_args['orderby'] = 'rand';
				break;
			case 'sales':
				$query_args['meta_key'] = 'total_sales'; // WPCS: slow query ok.
				$query_args['orderby']  = 'meta_value_num';
				break;
			case 'rating':
				$query_args['meta_key'] = '_wc_average_rating'; // WPCS: slow query ok.
				$query_args['orderby']  = 'meta_value_num';
				break;
			default:
				$query_args['orderby'] = 'date';
		}

		$p_query = new WP_Query( $query_args );

		add_filter('loop_shop_columns', array($this,'loop_columns'), 999);

		if ( $p_query->have_posts() ) :
			print '<div class="woocommerce e_brayn_woo_products">';
			print '<ul class="products columns-'. esc_attr( absint($column) ) .'">';

				while ( $p_query->have_posts() ) : $p_query->the_post();

					wc_get_template_part( 'content', 'product' );

				endwhile;

			print '</ul>';
			print '</div>';

		else :

			print '<h4>'. esc_html__( 'No products found!', 'brayn-core' ).'</h4>';

		endif;

		remove_filter('loop_shop_columns', array($this,'loop_columns'), 999);
		wp_reset_postdata();
		
	}
}
