<?php
/**
 * Action hooks doing by the theme
 *
 * @package WordPress
 * @subpackage Brayn
 * @since 1.0
 */

/**
 * Register the required plugins for this theme.
 *
 * This function is hooked into `tgmpa_register`, which is fired on the WP `init` action on priority 10.
 */
function brayn_register_required_plugins() {
	/*
	 * Array of plugin arrays. Required keys are name and slug.
	 * If the source is NOT from the .org repo, then source is also required.
	 */
	$plugins = array(
		array(
			'name'               => 'Brayn Core',
			'slug'               => 'brayn-core',
			'source'             => get_parent_theme_file_path( '/inc/tgmpa/plugins/brayn-core.zip' ),
			'required'           => true,
			'version'            => '2.0.0',
			'force_activation'   => false,
			'force_deactivation' => false,
			'external_url'       => '',
			'is_callable'        => '',
		),
		array(
			'name'     => 'Elementor Page Builder',
			'slug'     => 'elementor',
			'required' => true,
		),
		array(
			'name'     => 'WooCommerce',
			'slug'     => 'woocommerce',
			'required' => false,
		),
		array(
			'name'     => 'Elementor - Header, Footer & Blocks',
			'slug'     => 'header-footer-elementor',
			'required' => false,
		),
		array(
			'name'     => 'One Click Demo Import',
			'slug'     => 'one-click-demo-import',
			'required' => false,
		),
	);

	$config = array(
		'id'           => 'brayn',
		'default_path' => '',
		'menu'         => 'tgmpa-install-plugins',
		'has_notices'  => true,
		'dismissable'  => true,
		'dismiss_msg'  => '',
		'is_automatic' => false,
		'message'      => '',
	);

	tgmpa( $plugins, $config );
}
add_action( 'tgmpa_register', 'brayn_register_required_plugins' );

/**
 * Enqueue styles and scripts
 */
function brayn_enqueue_assets() {
	// fonts.
	wp_enqueue_style( 'brayn-fonts', brayn_font_families(), array(), '1.0' );
	// css files.
	wp_enqueue_style( 'bootstrap', esc_url( get_theme_file_uri( '/assets/lib/bootstrap/css/bootstrap.min.css' ) ), array(), '4.1.3' );
	wp_enqueue_style( 'font-awesome', esc_url( get_theme_file_uri( '/assets/lib/font-awesome/css/font-awesome.min.css' ) ), array(), '4.7.0' );
	wp_enqueue_style( 'brayn-stylesheet', esc_url( get_theme_file_uri( '/assets/css/site.css' ) ), array( 'bootstrap' ), filemtime( get_parent_theme_file_path( '/assets/css/site.css' ) ) );

	brayn_get_custom_skin_css();
	
	// mediaelement.
	if ( is_single() ) {
		wp_enqueue_style( 'wp-mediaelement' );
		wp_enqueue_script( 'wp-mediaelement', false, array( 'jquery' ), false, true );
	}

	// scripts.
	wp_enqueue_script( 'jquery-masonry' );
	wp_enqueue_script( 'smooth-scrollbar', esc_url( get_theme_file_uri( '/assets/js/smooth-scrollbar.js' ) ), array( 'jquery' ), '1.0.1', true );
	wp_enqueue_script( 'anime', esc_url( get_theme_file_uri( '/assets/js/anime.min.js' ) ), array( 'jquery' ), '3.0.1', true );
	wp_enqueue_script( 'appear', esc_url( get_theme_file_uri( '/assets/js/appear.min.js' ) ), array( 'jquery' ), '1.0.0', true );
	wp_enqueue_script( 'jarallax', esc_url( get_theme_file_uri( '/assets/js/jarallax.min.js' ) ), array( 'jquery' ), '1.10.6', true );
	wp_enqueue_script( 'magnific-popup', esc_url( get_theme_file_uri( '/assets/js/magnific-popup.min.js' ) ), array( 'jquery' ), '1.1.0', true );
	wp_enqueue_script( 'pixi', esc_url( get_theme_file_uri( '/assets/js/pixi.min.js' ) ), array( 'jquery' ), '4.8.5', true );
	wp_enqueue_script( 'swiper', esc_url( get_theme_file_uri( '/assets/js/swiper.min.js' ) ), array( 'jquery' ), '5.4.5', true );
	wp_enqueue_script( 'tilt-js', esc_url( get_theme_file_uri( '/assets/js/vanilla-tilt.min.js' ) ), array( 'jquery' ), '1.0.0', true );
	wp_enqueue_script( 'brayn-lib', esc_url( get_theme_file_uri( '/assets/js/app-lib.js' ) ), array( 'jquery', 'anime' ), '1.0.0', true );
	wp_enqueue_script( 'brayn-js', esc_url( get_theme_file_uri( '/assets/js/app.js' ) ), array( 'jquery', 'anime', 'brayn-lib' ), filemtime( get_parent_theme_file_path( '/assets/js/app.js' ) ), true );

	wp_localize_script(
		'brayn-js',
		'braynLib',
		[
			'filterObj' => esc_js( get_theme_file_uri( '/assets/images/clouds-s.jpg' ) ),
		]
	);

	// threaded comment script.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'brayn_enqueue_assets' );

if ( function_exists( 'brayn_core_add_custom_post_type' ) ) :
	/**
	 * Add portfolio custom post type
	 *
	 * @since 1.0
	 */
	brayn_core_add_custom_post_type(
		'portfolio',
		array(
			'labels'              => array(
				'name'               => esc_html__( 'Portfolios', 'brayn' ),
				'singular_name'      => esc_html__( 'Portfolio', 'brayn' ),
				'menu_name'          => _x( 'Portfolios', 'Admin menu name', 'brayn' ),
				'all_items'          => esc_html__( 'All Portfolios', 'brayn' ),
				'add_new'            => esc_html__( 'Add New', 'brayn' ),
				'add_new_item'       => esc_html__( 'Add New Portfolio', 'brayn' ),
				'edit'               => esc_html__( 'Edit', 'brayn' ),
				'edit_item'          => esc_html__( 'Edit Portfolio', 'brayn' ),
				'new_item'           => esc_html__( 'New Portfolio', 'brayn' ),
				'view'               => esc_html__( 'View Portfolio', 'brayn' ),
				'view_item'          => esc_html__( 'View Portfolio', 'brayn' ),
				'search_items'       => esc_html__( 'Search Portfolios', 'brayn' ),
				'not_found'          => esc_html__( 'No portfolios found', 'brayn' ),
				'not_found_in_trash' => esc_html__( 'No portfolios found in trash', 'brayn' ),
				'parent'             => esc_html__( 'Parent portfolio', 'brayn' )
				),
			'public'              => true,
			'show_ui'             => true,
			'capability_type'     => 'post',
			'map_meta_cap'        => true,
			'publicly_queryable'  => true,
			'exclude_from_search' => false,
			'menu_position'       => 5,
			'hierarchical'        => false,
			'rewrite'             => array( 'slug' => _x( 'portfolio', 'slug', 'brayn' ), 'with_front' => false, 'feeds' => true ),
			'query_var'           => true,
			'supports'            => array( 'title', 'editor', 'revisions', 'thumbnail' ),
			'has_archive'         => 'portfolios',
			'show_in_rest'        => true,
			'show_in_nav_menus'   => true,
			'menu_icon'           => 'dashicons-media-interactive'
		)
	);

	/**
	 * Add custom taxonomy
	 *
	 * @since 1.0
	 */
	brayn_core_add_custom_taxonomies(
		'portfolio-cat',
		array(
			'post_type_for' => 'portfolio',
			'ui' => array(
				array( 'portfolio' ),
				array(
					'hierarchical' => true,
					'label' => esc_html__( 'Portfolio Categories', 'brayn' ),
					'labels' => array(
						'name'              => esc_html__( 'Portfolio Categories', 'brayn' ),
						'singular_name'     => esc_html__( 'Portfolio Category', 'brayn' ),
						'menu_name'         => _x( 'Categories', 'Admin menu name', 'brayn' ),
						'search_items'      => esc_html__( 'Search Portfolio Categories', 'brayn'),
						'all_items'         => esc_html__( 'All Portfolio Categories', 'brayn' ),
						'parent_item'       => esc_html__( 'Parent Portfolio Category', 'brayn' ),
						'parent_item_colon' => esc_html__( 'Parent Portfolio Category:', 'brayn' ),
						'edit_item'         => esc_html__( 'Edit Portfolio Category', 'brayn' ),
						'update_item'       => esc_html__( 'Update Portfolio Category', 'brayn' ),
						'add_new_item'      => esc_html__( 'Add New Portfolio Category', 'brayn' ),
						'new_item_name'     => esc_html__( 'New Portfolio Category Name', 'brayn' ),
	            	),
	            	'show_ui' => true,
	            	'query_var' => true,
	            	'show_admin_column' => true,
	            	'show_in_rest' => true,
	            	'rewrite' => array(
	            		'slug' => _x( 'portfolio-category', 'slug', 'brayn' ),
	            		'with_front' => false,
	            		'hierarchical' => true,
	            	),
	            ),
			),
		)
	);
endif;

/**
 * Handles JavaScript detection.
 *
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 *
 * @since Brayn 1.0
 */
function brayn_javascript_detection() {
	echo "<script>(function(html){'use strict';html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
}
add_action( 'wp_head', 'brayn_javascript_detection', 0 );

/**
 * Register custom metabox.
 * Need the core plugin to be active
 *
 * @return void
 */
function brayn_register_page_metabox() {
	if ( class_exists( 'Brayn_Core_Metabox' ) ) {
		$option = array(
			'id'        => 'brayn_page_metabox',
			'post_type' => 'page',
			'title'     => esc_html__( 'Additional Settings', 'brayn' ),
			'options'   => array(
				'_brayn_page_subtitle' => array(
					'type'  => 'textarea',
					'args' => array(
						'id'          => '_brayn_page_subtitle',
						'label'       => esc_html__( 'Subtitle', 'brayn' ),
						'description' => esc_html__( 'Subtitle. Short and sweet.', 'brayn' ),
						'default'     => '',
					),
				),
				'elementor_hide_title' => array(
					'type' => 'elementor_hide_title',
					'args' => array(
						'id'          => 'elementor_hide_title',
						'label'       => esc_html__( 'Hide Title &amp; Subtitle', 'brayn' ),
						'description' => '',
						'default'     => '',
					),
				),
			),
		);

		new Brayn_Core_Metabox( $option );
	}
}
add_action( 'init', 'brayn_register_page_metabox' );

/**
 * Register custom metabox.
 * Need the core plugin to be active
 *
 * @return void
 */
function brayn_register_post_metabox() {
	if ( class_exists( 'Brayn_Core_Metabox' ) ) {
		$option = array(
			'id'        => 'brayn_post_metabox',
			'post_type' => 'post',
			'title'     => esc_html__( 'Additional Settings', 'brayn' ),
			'options'   => array(
				'_brayn_post_layout' => array(
					'type' => 'select',
					'args' => array(
						'id'          => '_brayn_post_layout',
						'label'       => esc_html__( 'Layout', 'brayn' ),
						'description' => '',
						'default'     => 'no-sidebar',
						'options'     => [
							'no-sidebar'    => esc_html__( 'Fullwidth', 'brayn' ),
							'left-sidebar'  => esc_html__( 'Left Sidebar', 'brayn' ),
							'right-sidebar' => esc_html__( 'Right Sidebar', 'brayn' ),
						],
					),
				),
			),
		);

		new Brayn_Core_Metabox( $option );
	}
}
add_action( 'init', 'brayn_register_post_metabox' );

/**
 * Filtering the custom logo output.
 *
 * @since version 1.0
 */
function _brayn_get_custom_logo( $html, $blog_id ) {
	if ( '' === $html ) {
		$html = sprintf( '<a href="%1$s" class="custom-logo-link brand" rel="home">%2$s</a>',
			esc_url( home_url( '/' ) ),
			'<img src="' . get_theme_file_uri( '/assets/images/logo.png' ) . '" srcset="' . get_theme_file_uri( '/assets/images/logo-2x.png' ) . ' 2x" class="custom-logo" alt="' . esc_attr( get_bloginfo( 'description' ) ) . '" />'
		);
	} elseif ( is_customize_preview() ) {
		$custom_logo_id = get_theme_mod( 'custom_logo' );
		if ( ! $custom_logo_id ) {
			$html = sprintf( '<a href="%1$s" class="custom-logo-link brand" rel="home">%2$s</a>',
				esc_url( home_url( '/' ) ),
				'<img src="' . get_theme_file_uri( '/assets/images/logo.png' ) . '" srcset="' . get_theme_file_uri( '/assets/images/logo-2x.png' ) . ' 2x" class="custom-logo" alt="' . esc_attr( get_bloginfo( 'description' ) ) . '" />'
			);
		} else {
			$image = wp_get_attachment_url( $custom_logo_id );
			$html = sprintf( '<a href="%1$s" class="custom-logo-link brand" rel="home">%2$s</a>',
				esc_url( home_url( '/' ) ),
				'<img src="' . esc_url( $image ) . '" class="custom-logo" alt="' . esc_attr( get_bloginfo( 'description' ) ) . '" />'
			);
		}
	} else {
		$custom_logo_id = get_theme_mod( 'custom_logo' );
		$image = wp_get_attachment_url( $custom_logo_id );
		$html = sprintf( '<a href="%1$s" class="custom-logo-link brand" rel="home">%2$s</a>',
			esc_url( home_url( '/' ) ),
			'<img src="' . esc_url( $image ) . '" class="custom-logo" alt="' . esc_attr( get_bloginfo( 'description' ) ) . '" />'
		);	
	}

	return $html;
}
add_filter( 'get_custom_logo', '_brayn_get_custom_logo', 10, 2 );

if ( ! function_exists( 'brayn_place_searchform' ) ) {
	/**
	 * Adds a searchform to the footer
	 * Javascript will call this elements on search button click
	 *
	 * @since version 1.0
	 */
	function brayn_place_searchform() {
		$placeholder = esc_html__( 'Search &hellip;', 'brayn' );
		if ( BRAYN_WOOCOMMERCE_ACTIVE && '' !== get_theme_mod( 'enable_woo_search', true ) ) {
			$placeholder = esc_html__( 'Search products &hellip;', 'brayn' );
		}

 		$form = '<form role="search" method="get" class="search-form" action="' . esc_url( home_url( '/' ) ) . '">
                <label><span class="screen-reader-text">' . esc_html__( 'Search for:', 'brayn' ) . '</span><input type="search" class="search-field" placeholder="' . esc_attr( $placeholder ) . '" value="' . get_search_query() . '" name="s" /></label>
                <input type="submit" class="search-submit" value="' . esc_attr__( 'Search', 'brayn' ) . '" />';
        if ( BRAYN_WOOCOMMERCE_ACTIVE && '' !== get_theme_mod( 'enable_woo_search', true ) ) {
        	$form .= '<input type="hidden" name="post_type" value="product">';
        }
        $form .= '</form>';

		print '<div id="brayn-search-form"><div class="brayn--search">' . $form . '<a class="brayn--close-search-form" href="#"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M278.6 256l68.2-68.2c6.2-6.2 6.2-16.4 0-22.6-6.2-6.2-16.4-6.2-22.6 0L256 233.4l-68.2-68.2c-6.2-6.2-16.4-6.2-22.6 0-3.1 3.1-4.7 7.2-4.7 11.3 0 4.1 1.6 8.2 4.7 11.3l68.2 68.2-68.2 68.2c-3.1 3.1-4.7 7.2-4.7 11.3 0 4.1 1.6 8.2 4.7 11.3 6.2 6.2 16.4 6.2 22.6 0l68.2-68.2 68.2 68.2c6.2 6.2 16.4 6.2 22.6 0 6.2-6.2 6.2-16.4 0-22.6L278.6 256z"/></svg></a></div></div>';
	}
}
add_action( 'wp_footer', 'brayn_place_searchform' );

/**
 * Filter the excerpt "read more" string.
 *
 * @param string $more "Read more" excerpt string.
 * @return string (Maybe) modified "read more" excerpt string.
 */
function brayn_excerpt_more( $more ) {
	return '&hellip;';
}
add_filter( 'excerpt_more', 'brayn_excerpt_more' );

/**
 * Filter the excerpt length.
 *
 * @param int $length default excerpt length.
 * @return int new excerpt length.
 */
function brayn_excerpt_length( $length ) {
	return 27;
}
add_filter( 'excerpt_length', 'brayn_excerpt_length', 999 );

/**
 * Modifies tag cloud widget arguments to display all tags in the same font size
 * and use list format for better accessibility.
 *
 * @since 1.0
 *
 * @param array $args Arguments for tag cloud widget.
 * @return array The filtered arguments for tag cloud widget.
 */
function brayn_widget_tag_cloud_args( $args ) {
	$args['largest']  = 1;
	$args['smallest'] = 1;
	$args['unit']     = 'em';
	$args['format']   = 'list';

	return $args;
}
add_filter( 'widget_tag_cloud_args', 'brayn_widget_tag_cloud_args' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 * @return void
 */
function brayn_theme_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar Archive', 'brayn' ),
			'id'            => 'sidebar-archive',
			'description'   => esc_html__( 'Sidebar for post archive pages. Add widgets here to appear in your sidebar.', 'brayn' ),
			'before_widget' => '<div id="%1$s" class="sidebarbox %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		)
	);

	if ( class_exists( 'Brayn_Core' ) ) {
		register_sidebar(
			array(
				'name'          => esc_html__( 'Single Post', 'brayn' ),
				'id'            => 'sidebar-single-post',
				'description'   => esc_html__( 'Sidebar for single post page. Add widgets here to appear in your sidebar.', 'brayn' ),
				'before_widget' => '<div id="%1$s" class="sidebarbox %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h4 class="widget-title">',
				'after_title'   => '</h4>',
			)
		);
	}

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 1', 'brayn' ),
			'id'            => 'footer-col-one',
			'description'   => esc_html__( 'Add widgets here to appear in footer column 1.', 'brayn' ),
			'before_widget' => '<div id="%1$s" class="sidebarbox br-widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 2', 'brayn' ),
			'id'            => 'footer-col-two',
			'description'   => esc_html__( 'Add widgets here to appear in footer column 2.', 'brayn' ),
			'before_widget' => '<div id="%1$s" class="sidebarbox br-widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Column 3', 'brayn' ),
			'id'            => 'footer-col-three',
			'description'   => esc_html__( 'Add widgets here to appear in footer column 3.', 'brayn' ),
			'before_widget' => '<div id="%1$s" class="sidebarbox br-widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h4 class="widget-title">',
			'after_title'   => '</h4>',
		)
	);

	if ( BRAYN_WOOCOMMERCE_ACTIVE ) {
		register_sidebar(
			array(
				'name'          => esc_html__( 'Sidebar Shop', 'brayn' ),
				'id'            => 'sidebar-shop',
				'description'   => esc_html__( 'Sidebar for shop pages. Add widgets here to appear in your sidebar.', 'brayn' ),
				'before_widget' => '<div id="%1$s" class="sidebarbox %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h4 class="widget-title">',
				'after_title'   => '</h4>',
			)
		);		
	}

}
add_action( 'widgets_init', 'brayn_theme_widgets_init' );

/**
 * Some dynamic div handlers
 *
 * @since 1.0.0
 */
function brayn_stretched_container_fixes() {
	?>
<script>
jQuery(function(t){"use strict";var n=function(){var n=t("#main").width(),i=t(".container").width(),a=t(".post-content-contain").width(),e=(n-a)/2;if(t("#secondary").length)return!1;t(".alignfull").each(function(){var i=this,a=t(i).data("dstyle");void 0!==a&&t(i).attr("style",a),t(i).css({width:n+"px",maxWidth:"none",marginLeft:-1*e+"px"})}),t(".alignwide").each(function(){var n=this,s=t(n).data("dstyle");t(n).removeAttr("style"),void 0!==s&&t(n).attr("style",s),e>50&&t(n).css({width:i+"px",maxWidth:"none",marginLeft:(i-a)/2*-1+"px"})})};t(window).on("load",function(){!function(){if(t("#secondary").length)return!1;t(".alignfull").each(function(){var n=t(this).attr("style");t(this).data("dstyle",n)}),t(".alignwide").each(function(){var n=t(this).attr("style");t(this).data("dstyle",n)}),n()}()}).on("resize",function(){n()})});
</script>
	<?php
}
add_action( 'wp_footer', 'brayn_stretched_container_fixes' );

/**
 * Filters the comment field.
 *
 * @param array $fields The comment form fields.
 * @return array
 */
function brayn_comment_form_field( $fields ) {
	$replace_comment = esc_html__( 'Write a response', 'brayn' );

	$fields['comment_field'] = '<p class="comment-form-comment"><textarea id="comment" name="comment" cols="45" rows="8" placeholder="' . esc_attr( $replace_comment ) . '" aria-required="true"></textarea></p>';

	return $fields;
}
add_filter( 'comment_form_defaults', 'brayn_comment_form_field' );

/**
 * Adds author box info in single post
 *
 * @since 1.0.0
 */
function brayn_append_author_box() {
	if ( brayn_post_author_box() ) {
		if ( '' !== get_theme_mod( 'show_author_box', true ) || is_customize_preview() ) {
			// customizer placeholder, not really shown in actual site.
			if ( is_customize_preview() ) {
				$relhiden = ( '' === get_theme_mod( 'show_author_box', true ) ? 'display:none;' : '' );
				print '<div id="author_box_customization" style="' . esc_attr( $relhiden ) . '">';
			}
			print brayn_post_author_box();
			if ( is_customize_preview() ) {
				print '</div>';
			}
		}
	}
}
add_action( 'brayn_after_single_entry', 'brayn_append_author_box', 15 );

function brayn_add_site_loader() {
	if ( '' !== get_theme_mod( 'enable_preloader', true ) ) {
		print '<div id="transit_roler"><div class="trl"></div><div class="trr"></div><div class="trload"></div></div>' . "\n";
	}
}
add_action( 'brayn_before_site_el', 'brayn_add_site_loader', 10 );

/**
 * Register header & footer elementor location
 *
 * @since 1.0.2
 */
function brayn_register_elementor_locations( $elementor_theme_manager ) {
	$elementor_theme_manager->register_location( 'header' );
	$elementor_theme_manager->register_location( 'footer' );
	$elementor_theme_manager->register_location( 'archive' );
	$elementor_theme_manager->register_location( 'single' );
}
add_action( 'elementor/theme/register_locations', 'brayn_register_elementor_locations' );

/**
 * Elementor Canvas Template Fixes
 *
 * @since 1.0.2
 */
function brayn_fix_elementor_canvas_before() {
	do_action( 'brayn_before_site_el' );
	print '<div id="site" class="page">';
}
add_action( 'elementor/page_templates/canvas/before_content', 'brayn_fix_elementor_canvas_before', 1 );

/**
 * Elementor Canvas Template Fixes
 *
 * @since 1.0.2
 */
function brayn_fix_elementor_canvas_before_content() {
	remove_all_actions( 'elementor/page_templates/canvas/before_content' );
	print '<div class="viewport">';
}
add_action( 'elementor/page_templates/canvas/before_content', 'brayn_fix_elementor_canvas_before_content', 999 );

/**
 * Elementor Canvas Template Fixes
 *
 * @since 1.0.2
 */
function brayn_fix_elementor_canvas_after_content() {
	remove_all_actions( 'elementor/page_templates/canvas/after_content' );
	print '</div><!-- .viewport -->' . "\n";
	
	print '</div><!-- END #site -->' . "\n";
}
add_action( 'elementor/page_templates/canvas/after_content', 'brayn_fix_elementor_canvas_after_content', 999 );

/**
 * Adds helper actions if HFE installed
 *
 * @since 1.0.2
 */
if ( class_exists( 'Header_Footer_Elementor' ) ) {

	function brayn_hfe_header_option() {
		$hfe_method = get_option( 'hfe_compatibility_option', '1' );
		return $hfe_method;
	}

	/**
	 * Plugin break the theme layout, so let's fix them!!!
	 *
	 * @since 1.0.2
	 */
	function brayn_hfe_header_wrap() {
		if ( "1" === brayn_hfe_header_option() && hfe_header_enabled() ) {
			remove_all_actions( 'wp_body_open' );
			ob_start();
		}
	}
	add_action( 'wp_body_open',  'brayn_hfe_header_wrap', 999 );

	/**
	 * Removes all contents from HFE
	 * Put back the `brayn_before_site_el` action and opening div #site
	 *
	 * @since 1.0.2
	 */
	function brayn_hfe_header_wrap_after() {
		
		if ( "1" === brayn_hfe_header_option() && hfe_header_enabled() ) {
			ob_end_clean();
			do_action( 'brayn_before_site_el' );
			print '<div id="site" class="page">';
		}

	}
	add_action( 'hfe_header',  'brayn_hfe_header_wrap_after', 99 );

	/**
	 * Call the HFE header content and Put back the `viewport` opening div
	 *
	 * @since 1.0.2
	 */
	function brayn_hfe_after_header_wrap() {
		if ( "1" === brayn_hfe_header_option() && hfe_header_enabled() ) {

			hfe_render_header();
			print '<div class="viewport">';
			remove_all_actions( 'hfe_header' );
		}
	}
	add_action( 'hfe_header',  'brayn_hfe_after_header_wrap', 100 );

	/**
	 * Put back the closes divs
	 *
	 * @since 1.0.2
	 */
	function brayn_hfe_after_footer() {
		if ( "1" === brayn_hfe_header_option() && hfe_footer_enabled() ) {
			remove_all_actions( 'hfe_footer' );
			print '</div><!-- .viewport -->' . "\n";
			
			print '</div><!-- END #site -->' . "\n";
		}
	}
	add_action( 'hfe_footer',  'brayn_hfe_after_footer', 999 );
}

/**
 * Inject new classes to the body tag
 *
 * @since 2.0.0
 */
if ( ! function_exists( 'braynwp_body_classes' ) ) {

	function braynwp_body_classes( $classes ) {
		global $wp_query;

		if ( BRAYN_WOOCOMMERCE_ACTIVE ) {
			if ( is_singular('product') ) {
				$layout = brayn_get_single_product_layout();
				$classes[] = 'brayn-layout-' . trim($layout);
			}
		}

		return $classes;
	}

	add_filter( 'body_class', 'braynwp_body_classes' );

}

/**
 * Header button. 
 *
 * @since 2.0.0
 */
function brayn_header_complementary_button() {
	if ( get_theme_mod( 'enable_header_button', false ) || is_customize_preview() ) {
		$attr_target = get_theme_mod( 'header_button_url_target', false ) ? '_blank' : '_self';
		$attr_style  = is_customize_preview() ? ( get_theme_mod( 'enable_header_button', false ) ? '' : 'display:none;' ) : '';

		print '<a href="' . esc_url( get_theme_mod( 'header_button_url', '#' ) ) . '" class="button button-to-quote" target="' . esc_attr(  $attr_target ) . '" style="' . esc_attr( $attr_style ) . '">' . wp_kses_post( get_theme_mod( 'header_button_text', esc_html__( 'Button Text', 'brayn' ) ) ) . '</a>';
	}
}
add_action( 'brayn_header_complementary_content', 'brayn_header_complementary_button', 10 );

/**
 * Header search. 
 *
 * @since 2.0.0
 */
function brayn_header_complementary_search_button() {
	if ( '' !== get_theme_mod( 'enable_search', true ) || is_customize_preview() ) {
		$attr_style  = is_customize_preview() ? ( '' === get_theme_mod( 'enable_search', true ) ? 'display:none;' : '' ) : '';

		print '<a href="#" class="search-caller" style="' . esc_attr( $attr_style ) . '">' . brayn_get_icon_svg( 'search', 18 ) . '</a>';
	}
}
add_action( 'brayn_header_complementary_content', 'brayn_header_complementary_search_button', 20 );
