<?php
/**
 * Some helper functions
 *
 * @package Brayn
 * @since 1.0
 */

/**
 * Register custom fonts
 * credit to twentyseventeen theme code
 *
 * @since 1.0
 */
function brayn_font_families() {
	$heading_font = get_theme_mod( 'brayn_heading_font', WIP_THEMES_DEFAULT_HEADING_FONT );
	$body_font    = get_theme_mod( 'brayn_body_font', WIP_THEMES_DEFAULT_BODY_FONT );

	$font_families = array( $heading_font, $body_font );
	$font_families = array_unique( $font_families );
	$query_args    = array(
		'family' => implode( '|', $font_families ),
		'subset' => 'latin,latin-ext',
	);
	$fonts_url     = add_query_arg( $query_args, '//fonts.googleapis.com/css' );
	return esc_url_raw( $fonts_url );
}

/**
 * Set the elementor global settings
 *
 * @return void
 */
function brayn_set_elementor_globals() {
	if ( did_action( 'elementor/loaded' ) ) {
		// 1. stretched section.
		update_option( 'elementor_stretched_section_container', '#main' );
		// 2. page title selector.
		update_option( 'elementor_page_title_selector', 'header.page-header' );
		// 3. Tablet break point.
		update_option( 'elementor_container_width', '1240' );
		update_option( 'elementor_viewport_lg', '992' );
		// 4. Space between widgets.
		update_option( 'elementor_space_between_widgets', '30' );
	}
}

/**
 * Gets the SVG code for a given icon.
 *
 * @param string $icon The icon name.
 * @param int    $size The desired size.
 * @return string
 */
function brayn_get_icon_svg( $icon, $size = 24 ) {
	return Brayn_SVG_Icons::get_svg( 'ui', $icon, $size );
}

/**
 * Gets the SVG code for a given social icon.
 *
 * @param string $icon The icon name.
 * @param int    $size The desired size.
 * @return string
 */
function brayn_get_social_icon_svg( $icon, $size = 24 ) {
	return Brayn_SVG_Icons::get_svg( 'social', $icon, $size );
}

/**
 * Detects the social network from a URL and returns the SVG code for its icon.
 *
 * @param string $uri  URL.
 * @param int    $size The desired size.
 * @return string
 */
function brayn_get_social_link_svg( $uri, $size = 24 ) {
	return Brayn_SVG_Icons::get_social_link_svg( $uri, $size );
}

/**
 * Get google font lists
 *
 * @since 1.0
 */
function brayn_get_google_fonts_data() {
	if ( ! class_exists( 'Brayn_Core' ) ) {
		return false;
	}

	$fonts = brayn_core_get_google_fonts( 'all' );
	return $fonts;
}

/**
 * Load the skin css file
 *
 * @return void
 */
function brayn_get_custom_skin_css() {
	$def = esc_url( get_theme_file_uri( '/assets/css/default-skin.css' ) );
	$ver = filemtime( get_parent_theme_file_path( '/assets/css/default-skin.css' ) );
	if ( ! class_exists( 'Brayn_Core' ) ) {
		wp_enqueue_style( 'brayn-skin', $def, array(), $ver );
		return false;
	}

	$module = new Brayn_Skin_Loader();
	// System failed to write css, recorded.
	if ( get_theme_mod( 'brayn-failed-write-css' ) === 'yes' ) {	

		if ( $module->set_status() ) {
			
			$custom_inline_skin = get_theme_mod( 'brayn_custom_skin_code' );

			if ( $custom_inline_skin && '' !== $custom_inline_skin ) {
				wp_add_inline_style( 'brayn-stylesheet', $custom_inline_skin );
			} else {
				wp_enqueue_style( 'brayn-skin', $def, array(), $ver );
			}

		} else {
			// Custom skin not active, get typography skin.
			wp_enqueue_style( 'brayn-skin', $def, array(), $ver );

			$typograph = get_theme_mod( 'brayn_typography_skin_code' );
			if ( $typograph && '' !== $typograph ) {
				wp_add_inline_style( 'brayn-skin', $typograph );
			}			
		}
	} else {

		if ( $module->set_status() ) {
			$css_version = get_theme_mod( 'brayn-custom-css-version' );
			$css_url     = $module->get_css_url();

			if ( $css_version &&  '' !== $css_version ) {
				wp_enqueue_style( 'brayn-skin', esc_url( $css_url . '/brayn-custom-skin.css' ), array( 'brayn-stylesheet' ), $css_version );
			} else {
				wp_enqueue_style( 'brayn-skin', $def, array(), $ver );
			}
		} else {
			wp_enqueue_style( 'brayn-skin', $def, array(), $ver );

			$typograph = get_theme_mod( 'brayn_typography_skin_code' );
			if ( $typograph && '' !== $typograph ) {
				wp_add_inline_style( 'brayn-skin', $typograph );
			}
		}
	}
}

/**
 * The tricky theme requirements, font and style need 100% same (or close) with front-end
 * we will update the editor-style.css everytime user change the font and skin in customizer
 * need the core plugin to be activate
 *
 * @since 1.0
 */
function brayn_write_editor_style_css() {
	$cache_time_db = get_theme_mod( 'brayn_editor_cache_action', false );

	// Plugin is not active? leave the current editor style.
	if ( ! class_exists( 'Brayn_Core' ) ) {
		return false;
	}

	// No updates? stop here.
	if ( $cache_time_db ) {
		return false;
	}

	$module = new Brayn_Skin_Loader();
	$css    = $module->get_compiled_editor_style_css();
	brayn_core_write_to_file( get_parent_theme_file_path( '/assets/css/editor-style.css' ), $css );

	// Lock it, prevent any looping action.
	set_theme_mod( 'brayn_editor_cache_action', intval( time() ) );
}
add_action( 'admin_init', 'brayn_write_editor_style_css' );

/**
 * Get the first post in loop
 *
 * @since 1.0
 */
function brayn_get_first_post_loop() {
	// User setting defined.
	if ( 'no' === brayn_show_first_post_as_featured() ) {
		return false;
	}

	if ( ! is_home() ) {
		return false;
	}

	// Read the current page.
	$paged = ( get_query_var('paged') ) ? get_query_var('paged') : 1;
	if ( $paged < 1 ) {
		$paged = 1;
	}

	// Stop here, if we're on page > 1.
	if ( $paged > 1 ) {
		return false;
	}

	$fQuery = new WP_Query('showposts=1');
	
	if ( $fQuery->have_posts() ) {
		print '<div class="br-section py-0">' . "\n";
		$a = 0;
		while ( $fQuery->have_posts() ) {
			$fQuery->the_post();

			if ( $a > 0 ) {
				continue;
			}
			get_template_part( 'template-parts/post/loop', 'first' );

			$a++;
		}

		wp_reset_postdata();

		print '</div>' . "\n";

		unset($a);
	}
}
add_action( 'brayn_before_post_loop', 'brayn_get_first_post_loop', 10 );

/**
 * Show the first post as featured post
 *
 * @since 1.0
 */
function brayn_show_first_post_as_featured() {
	$default = true;
	$option = get_theme_mod( 'brayn_firstpost_featured', $default );

	if ( is_category() || is_tag() || is_author() || is_year() || is_month() || is_day() || is_tax() ) {
		$option = false;
	}

	$val = ( ! $option || '' === $option ) ? 'no' : 'yes';

	return $val;
}

/**
 * Get index layout
 *
 * @since 1.0
 */
function brayn_get_index_layout() {
	$default = 'no-sidebar';
	$option = get_theme_mod( 'brayn_index_layout', $default );

	return $option;
}

/**
 * Get page layout
 *
 * @since 1.0
 */
function brayn_get_page_layout() {
	global $post;

	$default = 'no-sidebar';
	$option = get_post_meta( $post->ID, '_brayn_post_layout', true );
	if ( ! $option ) {
		$option = $default;
	}

	return $option;
}

/**
 * Get index loop style
 *
 * @since 1.0
 * @return string 	{column | column-minimal | staggered | classic}
 */
function brayn_get_post_loop_style() {
	$default = 'column';
	$option = get_theme_mod( 'brayn_index_loop_style', $default );

	return $option;
}

/**
 * Get term loop style
 *
 * @since 1.0
 * @return string 	{column | column-minimal | staggered | classic}
 */
function brayn_get_tax_post_loop_style() {
	global $wp_query;

	$term_id = get_queried_object_id();

	if ( $term_id ) {
		$style = get_term_meta( $term_id, '_brayn_core_term_listing_style', true );
		if ( ! $style ) {
			$style = brayn_get_post_loop_style();
		}

		return $style;
	}

	return brayn_get_post_loop_style();
}

/**
 * Get term layout
 *
 * @since 1.0
 * @return string 	{column | column-minimal | staggered | classic}
 */
function brayn_get_term_layout() {
	global $wp_query;

	$term_id = get_queried_object_id();

	if ( $term_id ) {
		$layout = get_term_meta( $term_id, '_brayn_core_term_layout', true );
		if ( ! $layout ) {
			$layout = brayn_get_index_layout();
		}

		return $layout;
	}

	return brayn_get_index_layout();
}

if ( ! function_exists( 'brayn_get_single_post_layout' ) ) :
	/**
	 * Get single post layout
	 *
	 * @since 1.0.0
	 */
	function brayn_get_single_post_layout() {
		global $post;

		$default = 'no-sidebar';
		$option = get_post_meta( $post->ID, '_brayn_post_layout', true );
		if ( ! $option ) {
			$option = $default;
		}

		return $option;
	}
endif;

/**
 * get previous/next post thumbnail
 * used within the loop
 *
 * @global $post 	Single post object
 * @return mixed 	string thumbnail || bool when failure
 */
function brayn_get_prev_next_post_thumbnail( $dir = 'prev', $size = 'large' ) {
	global $post;

	$the_post = ( $dir === 'prev' ) ? get_previous_post() : get_next_post();
	if ( empty( $the_post ) ) {
		return false;
	}

	if ( has_post_thumbnail( $the_post ) ) {
		return get_the_post_thumbnail( $the_post, $size );
	} else {
		return '<span class="no-thumb-svg">' . brayn_get_icon_svg( 'doc', 24 ) . '</span>';
	}
}

function brayn_page_title_is_hidden( $postID ) {
	global $post;

	// get the page settings.
	$settings = get_post_meta( $postID, '_elementor_page_settings', true );
	if ( ! $settings || null === $settings ) {
		return false;
	}

	if ( isset( $settings['hide_title'] ) && "yes" === $settings['hide_title'] ) {
		return true;
	}

	return false;
}

/**
 * Check whether post is edited using elementor
 *
 * @return bool
 */
function brayn_post_is_built_with_elementor( $postID ) {
	if ( ! did_action( 'elementor/loaded' ) ) {
		return false;
	}

	return \Elementor\Plugin::$instance->db->is_built_with_elementor( $postID );
}

/**
 * Get active social links and icons
 * based on user settings
 *
 * @since 1.0
 */
function brayn_print_active_social_links() {
	$settings = array(
		'social_dribbble',
		'social_facebook',
		'social_github',
		'social_medium',
		'social_instagram',
		'social_linkedin',
		'social_pinterest',
		'social_twitter',
		'social_youtube',
	);

	$output = '';
	foreach ( $settings as $social ) {
		$link = get_theme_mod( $social, '' );

		if ( is_customize_preview() ) {
			$hideit = ( '' === trim( $link ) ? ' hide-me' : '' );
			
			$output .= '<li class="' . esc_attr( $social ) . esc_attr( $hideit ) . '"><a href="' . esc_url( $link ) . '">' . brayn_get_social_icon_svg( str_replace( 'social_', '', $social ), 20 ) . '</a></li>';
		} else {
			if ( trim( $link ) !== '' ) {
				$output .= '<li class="' . esc_attr( $social ) . '"><a href="' . esc_url( $link ) . '">' . brayn_get_social_icon_svg( str_replace( 'social_', '', $social ), 20 ) . '</a></li>';
			}
		}
	}

	$kses_defaults = wp_kses_allowed_html( 'post' );
	$svg_args      = array(
		'svg'   => array(
			'class'           => true,
			'aria-hidden'     => true,
			'aria-labelledby' => true,
			'role'            => true,
			'xmlns'           => true,
			'width'           => true,
			'height'          => true,
			'viewbox'         => true, // <= Must be lower case!
		),
		'g'     => array( 'fill' => true ),
		'title' => array( 'title' => true ),
		'path'  => array(
			'd'    => true,
			'fill' => true,
		),
		'ellipse' => array(
			'transform' => true,
			'cx'        => true,
			'cy'        => true,
			'rx'        => true,
			'ry'        => true,
		),
		'li' => array(
			'id' => true,
			'class' => true,
			'style' => array('display' => true ),
		),
	);
	$allowed_tags  = array_merge( $kses_defaults, $svg_args );

	print wp_kses( $output, $allowed_tags );
}

function brayn_page_title_container_classes() {
	global $post;
	$classes = ['container'];
	if ( ! brayn_post_is_built_with_elementor( get_the_ID() ) ) {
		$classes[] = 'text-center';
	}
	$class = 'class="' . implode( ' ', $classes ) . '"';
	print wp_kses( $class, array( "\'", '\"' ) );
}
