<?php
/**
 * Custom template tags for this theme
 *
 * @package WordPress
 * @subpackage Brayn
 * @since 1.0.0
 */

if ( ! function_exists( 'brayn_category_lists' ) ) :
	/**
	 * get post category
	 */
	function brayn_category_lists() {
		global $post;
		$categories_list = get_the_category_list( esc_html__( ' &#47; ', 'brayn' ) );

		$stick = '';
		if ( ! is_single() && is_sticky() ) {
			$stick = '<span class="stick-ol">
				<svg viewBox="0 0 337.162 337.162" style="enable-background:new 0 0 337.162 337.162;" xml:space="preserve">
				<path d="M331.314,105.653L231.612,5.951c-3.766-3.766-8.789-5.84-14.143-5.84c-5.354,0-10.376,2.074-14.142,5.84l-20.506,20.506
					c-7.798,7.798-7.797,20.486,0,28.284l1.817,1.817l-73.825,63.353l-7.2-7.2c-3.911-3.911-9.453-6.154-15.205-6.154
					c-5.064,0-9.917,1.718-13.664,4.838L43.009,137.82c-4.16,3.463-6.582,8.261-6.821,13.509c-0.239,5.248,1.737,10.246,5.564,14.073
					l52.547,52.547c-18.038,18.14-43.306,43.606-46.531,47.132c-4.848,5.288-9.471,10.641-13.844,15.921
					c-8.744,10.564-16.398,20.93-22.571,30.202c-3.09,4.634-5.814,8.988-8.149,12.928c-7.116,12.45-1.695,16.457,9.576,9.576
					c3.94-2.335,8.294-5.06,12.928-8.149c9.272-6.173,19.638-13.828,30.202-22.572c5.28-4.373,10.633-8.996,15.921-13.844
					c3.526-3.225,28.992-28.493,47.132-46.531l52.901,52.901c3.6,3.6,8.303,5.583,13.243,5.583c5.494,0,10.72-2.493,14.339-6.84
					l26.423-31.736c6.903-8.29,6.313-21.24-1.315-28.869l-7.2-7.2l63.353-73.825l1.817,1.817c3.766,3.766,8.788,5.839,14.143,5.839
					c5.354,0,10.376-2.074,14.142-5.839l20.506-20.506C339.112,126.139,339.112,113.451,331.314,105.653z"/>
				</svg>' . esc_html__( 'Sticky', 'brayn' ) .
			'</span>';
		}
		if ( $categories_list ) {
			/* translators: 1: posted in label, only visible to screen readers. 2: Sticky status. 3: list of categories. */
			printf(
				'<span class="cat-links"><span class="screen-reader-text">%1$s</span>%2$s%3$s</span>',
				esc_html__( 'Posted in', 'brayn' ),
				$stick,
				$categories_list
			); // WPCS: XSS OK.
		}
	}
endif;

if ( ! function_exists( 'brayn_get_post_author' ) ) :
	/**
	 * Gets the post author
	 */
	function brayn_get_post_author()  {
		return '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . get_the_author() . '</a></span>';
	}
endif;

if ( ! function_exists( 'brayn_time_link' ) ) :
	/**
	 * Gets a nicely formatted string for the published date.
	 */
	function brayn_time_link() {
		$time_string = '<time class="entry-date published updated" datetime="%1$s"><span class="day">%2$s</span><span class="month-and-year">%3$s</span></time>';

		$time_string = sprintf(
			$time_string,
			get_the_date( DATE_W3C ),
			get_the_date( 'd' ),
			get_the_date( 'M y' )
		);

		// Wrap the time string in a link, and preface it with 'Posted on'.
		return sprintf(
			/* translators: %s: post date */
			__( '<span class="screen-reader-text">Posted on</span> %s', 'brayn' ),
			$time_string
		);
	}
endif;

if ( ! function_exists( 'brayn_option_time_link' ) ) :
	/**
	 * Gets a nicely formatted string for the published date.
	 */
	function brayn_option_time_link() {
		$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';

		$time_string = sprintf(
			$time_string,
			get_the_date( DATE_W3C ),
			get_the_date()
		);

		// Wrap the time string in a link, and preface it with 'Posted on'.
		return sprintf(
			/* translators: %s: post date */
			__( '<span class="screen-reader-text">Posted on</span> %s', 'brayn' ),
			$time_string
		);
	}
endif;

if ( ! function_exists( 'brayn_excerpt' ) ) {
	/**
	 * Custom excerpt length
	 */
	function brayn_excerpt($limit) {
		print wpautop( wp_trim_words( get_the_excerpt(), $limit ) );
	}
}

if ( ! function_exists( 'brayn_entry_tag' ) ) :
	/**
	 * Prints post tag(s)
	 */
	function brayn_entry_tag() {
		// Get Tags for posts.
		$tags_list = get_the_tag_list( '', ' ' );

		if ( $tags_list ) {

			if ( get_theme_mod( 'show_tag_lists', true ) !== '' || is_customize_preview() ) {
				// Customizer placeholder, not really shown in actual site.
				if ( is_customize_preview() ) {
					$relhiden = ( false === get_theme_mod( 'show_tag_lists', true ) ? 'display:none;' : '' );
					print '<div id="tag_lists_customization" style="' . esc_attr( $relhiden ) . '">';
				}

				echo '<div class="tag-links"><span class="screen-reader-text">' . esc_html__( 'Post tagged with:', 'brayn' ) . '</span>' .
				wp_kses(
					$tags_list,
					array(
						'a' => array(
							'href'  => array(),
							'title' => array(),
						),
					)
				) . '</div>';

				if ( is_customize_preview() ) {
					print '</div>';
				}
			}
		}
	}
endif;

if ( ! function_exists( 'brayn_post_author_box' ) ) :
	/**
	 * Display author box
	 *
	 * @since 1.0
	 * @return string
	 */
	function brayn_post_author_box() {
		global $post;
		if ( ! is_object( $post ) || ! is_single() ) {
			return false;
		}

		// no author for this post? don't bother.
		if ( ! isset( $post->post_author ) ) {
			return false;
		}

		$display_name = get_the_author_meta( 'display_name', $post->post_author );
		// use nickname instead if display_name is empty.
		if ( empty( $display_name ) ) {
			$display_name = get_the_author_meta( 'nickname', $post->post_author );
		}

		// description.
		$user_description = get_the_author_meta( 'user_description', $post->post_author );

		// no author description for this post? don't bother.
		if ( '' === $user_description ) {
			return false;
		}

		// posts archive link.
		$user_posts_link = get_author_posts_url( intval( get_the_author_meta( 'ID' , $post->post_author ) ) );

		// user website URL.
		$user_website = get_the_author_meta( 'url', $post->post_author );

		// user avatar.
		$user_avatar  = get_avatar( get_the_author_meta( 'user_email', $post->post_author ) , 90, '', $display_name );
		$user_socials = brayn_get_author_social( $post->post_author );
		$user_socials = ( false !== $user_socials ) ? $user_socials : '';

		$kses_defaults = wp_kses_allowed_html( 'post' );
		$svg_args      = array(
			'svg'   => array(
				'class'           => true,
				'aria-hidden'     => true,
				'aria-labelledby' => true,
				'role'            => true,
				'xmlns'           => true,
				'width'           => true,
				'height'          => true,
				'viewbox'         => true, // <= Must be lower case!
			),
			'g'     => array( 'fill' => true ),
			'title' => array( 'title' => true ),
			'path'  => array(
				'd'    => true,
				'fill' => true,
			),
		);
		$allowed_tags  = array_merge( $kses_defaults, $svg_args );

		$html = '<div class="post-author-box">' . "\n";
		$html .= '<h3 class="post-author-heading">' . esc_html__( 'About the author', 'brayn' ) . '</h3>' . "\n";
		$html .= '<div class="author-desc">' . "\n";
		$html .= $user_avatar;

		$html .= '<h6>' . esc_html( $display_name ) . '</h6>' . "\n";
		$html .= wp_kses( wpautop( $user_description ) . $user_socials, $allowed_tags );
		$html .= '</div>' . "\n";
		$html .= '</div>' . "\n";

		return $html;
	}
endif;

if ( ! function_exists( 'brayn_get_author_social' ) ) :
	/**
	 * Get author social links.
	 *
	 * @param int $user_id User database ID.
	 */
	function brayn_get_author_social( $user_id ) {
		$user_id = intval( $user_id );
		if ( ! $user_id || ! class_exists( 'Brayn_Core' ) ) {
			return false;
		}

		$socials = array(
			'_author_facebook',
			'_author_twitter',
			'_author_instagram',
			'_author_youtube',
			'_author_vimeo',
			'_author_google-plus',
			'_author_pinterest',
			'_author_medium',
			'_author_tumblr',
		);

		$kses_defaults = wp_kses_allowed_html( 'post' );
		$svg_args      = array(
			'svg'   => array(
				'class'           => true,
				'aria-hidden'     => true,
				'aria-labelledby' => true,
				'role'            => true,
				'xmlns'           => true,
				'width'           => true,
				'height'          => true,
				'viewbox'         => true, // <= Must be lower case!
			),
			'g'     => array( 'fill' => true ),
			'title' => array( 'title' => true ),
			'path'  => array(
				'd'    => true,
				'fill' => true,
			),
		);
		$allowed_tags  = array_merge( $kses_defaults, $svg_args );

		$html_link = '';
		foreach ( $socials as $social_id ) {
			$link = get_the_author_meta( esc_html( $social_id ), $user_id );
			if ( '' !== $link ) {
				$html_link .= '<a class="' . esc_attr( $social_id ) . '" href="' . esc_url( $link ) . '" rel="nofollow">' . wp_kses( brayn_get_social_icon_svg( str_replace( '_author_', '', $social_id ), 18 ), $allowed_tags ) . '</a>';
			}
		}
		unset( $social_id );

		if ( '' !== $html_link ) {
			return '<div class="author-social-links">' . $html_link . '</div>' . "\n";
		}

		return false;
	}
endif;

/**
 * Get portfolio category.
 *
 * @param int    $id        Post ID.
 * @param string $delimiter Delimiter string.
 */
function brayn_get_portfolio_cat_lists( $id, $delimiter = ', ' ) {
	$cats = get_the_term_list( $id, 'portfolio-cat', '', $delimiter );
	if ( $cats ) {
		return strip_tags( $cats );
	}
	return false;
}

/**
 * Get portfolio cover image.
 *
 * @global obj $post
 * @return string $image Html image markup
 */
function brayn_get_portfolio_cover_poster() {
	global $post;

	// Default.
	$image = esc_url( get_theme_file_uri( '/assets/images/portfolio-def-img.jpg' ) );
	if ( has_post_thumbnail() ) {
		$image = get_the_post_thumbnail( $post, 'brayn-post-potrait' );
	}
	return $image;
}

/**
 * Get portfolio cover image blur version.
 *
 * @global obj $post
 * @return string $image Html image markup
 */
function brayn_get_portfolio_cover_blurry_poster() {
	global $post;

	if ( function_exists( 'brayn_core_get_blurry_image' ) ) {
		brayn_core_get_blurry_image( $post );
	}

	return false;
}
