<?php
/**
 * Metaboxes register
 *
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( 'Brayn_Core_Metabox' ) ) {

	class Brayn_Core_Metabox {
		/**
		 * var $public
		 *
		 * @access public
		 * @since 1.0
		 */
		public $options = array();

		/**
		 * Brayn_Core_Metabox's instance
		 *
		 * @access public
		 * @since 1.0
		 * @return instance of Brayn_Core_Metabox
		 */
		public function __construct( $options = array() ) {
			if ( ! empty( $options ) ) {
				$this->options = $options;
				
				$this->init();
			}
		}

		public function init() {
			add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
			add_action( 'save_post', array( $this, 'save' ) );
		}

		public function add_meta_box() {
			if ( ! empty( $this->options ) ) {
				$id = $this->options['id'];
				$title = $this->options['title'];
				$post_type =$this->options['post_type'];
				
				add_meta_box( $id, $title, [ $this, 'metabox_ui' ], $post_type, 'side' );
			}
		}

		/**
		 * save the metabox value
		 *
		 * @access public
		 * @since 1.0
		 * @return void
		 */
		public function save( $post_id ) {
			$is_autosave = wp_is_post_autosave( $post_id );
			$is_revision = wp_is_post_revision( $post_id );

			if ( $is_autosave || $is_revision ) {
				return;
			}

			if ( empty( $this->options ) ) {
				return;
			}

			$options = $this->options['options'];
			foreach( $options as $id => $opt ) {
				if ( isset( $_POST[ trim($id) ] ) ) {
					if ( 'elementor_hide_title' === $id ) {
						$tosaved = [
							'hide_title' => 'yes',
						];
						update_post_meta( $post_id, '_elementor_page_settings', $tosaved );
					} else {
						update_post_meta( $post_id, trim($id), sanitize_text_field( $_POST[ trim($id) ] ) );
					}
				} else {
					if ( 'elementor_hide_title' === $id ) {

						//error_log( did_action( 'elementor/db/before_save' ) );

						if ( did_action( 'elementor/loaded' ) ) {
							if ( ! did_action( 'elementor/db/before_save' ) ) {
								$tosaved = [
									'hide_title' => '',
								];
								update_post_meta( $post_id, '_elementor_page_settings', $tosaved );
							}
						} else {
							$tosaved = [
								'hide_title' => '',
							];
							update_post_meta( $post_id, '_elementor_page_settings', $tosaved );
						}

					}
				}
			}
			unset( $opt );
		}

		/**
		 * register the metabox
		 *
		 * @access public
		 * @since 1.0
		 * @return void
		 */
		public function metabox_ui( $post ) {
			if ( empty( $this->options ) ) {
				return false;
			}
			$options = $this->options['options'];

			foreach( $options as $id => $opt ) {
				$ui = $this->ui( $post, $opt['type'], $opt['args'] );
				print $ui;
			}
			unset( $opt );
		}

		public function ui( $post, $type = '', $args = array() ) {
			if ( empty( $type ) || empty( $args ) ) {
				return false;
			}

			$interface = '';
			switch ($type) {
				case 'select':
					$interface = $this->ui_select( $post, $args );
					break;

				case 'elementor_hide_title':
					$interface = $this->ui_hide_title_elementor( $post, $args );
					break;

				case 'textarea':
					$interface = $this->ui_textarea( $post, $args );
					break;

				case 'text':
				default:
					$interface = $this->ui_text( $post, $args );
					break;
			}

			return $interface;
		}

		/**
		 * Text field
		 *
		 * @return string
		 */
		public function ui_text( $post, $args = array() ) {
			$args = wp_parse_args( $args,
				array(
					'id' => '',
					'label' => '',
					'description' => '',
					'default' => ''
				) );

			if ( empty( $args['id'] ) ) {
				return false;
			}

			$value = get_post_meta( $post->ID, $args['id'], true );
			if ( ! $value ) {
				$value = $args['default'];
			}

			$ui = '<div class="components-panel__row">' . "\n";
			$ui .= '<label for="' . esc_attr( $args['id'] ) . '">' . $args['label'] . '</label>';
			$ui .= '<div class="components-base-control"><div class="components-base-control__field">';

			$ui .= '<input type="text" name="' . esc_attr( $args['id'] ) . '" id="' . esc_attr( $args['id'] ) . '" value="' . esc_attr( $value ) . '">';

			$ui .= '</div></div>' . "\n";
			$ui .= '</div>' . "\n";

			if ( ! empty( $args['description'] ) ) {
				$ui .= '<div><small>' . stripslashes( $args['description'] ) . '</small></div>';
			}

			return $ui;
		}

		/**
		 * Text field
		 *
		 * @return string
		 */
		public function ui_textarea( $post, $args = array() ) {
			$args = wp_parse_args( $args,
				array(
					'id' => '',
					'label' => '',
					'description' => '',
					'default' => ''
				) );

			if ( empty( $args['id'] ) ) {
				return false;
			}

			$value = get_post_meta( $post->ID, $args['id'], true );
			if ( ! $value ) {
				$value = $args['default'];
			}

			$ui = '<div class="components-base-control">' . "\n";
			$ui .= '<div class="components-base-control__field">';

			$ui .= '<textarea name="' . esc_attr( $args['id'] ) . '" id="' . esc_attr( $args['id'] ) . '" row="3" placeholder="' . esc_attr( $args['label'] ) . '" style="width:100%;">' . esc_textarea( $value ) . '</textarea>';

			$ui .= '</div>' . "\n";
			$ui .= '</div>' . "\n";

			if ( ! empty( $args['description'] ) ) {
				$ui .= '<div><small>' . stripslashes( $args['description'] ) . '</small></div>';
			}

			return $ui;
		}

		/**
		 * Select option
		 *
		 * @return string
		 */
		public function ui_select( $post, $args = array() ) {
			$args = wp_parse_args( $args,
				array(
					'id' => '',
					'label' => '',
					'options' => array(),
					'description' => '',
					'default' => ''
				) );

			if ( empty( $args['id'] ) ) {
				return false;
			}

			$value = get_post_meta( $post->ID, $args['id'], true );
			if ( ! $value ) {
				$value = $args['default'];
			}

			$ui = '<div class="components-panel__row">' . "\n";
			$ui .= '<label for="' . esc_attr( $args['id'] ) . '">' . $args['label'] . '</label>';
			$ui .= '<div class="components-base-control"><div class="components-base-control__field">';

			$ui .= '<select name="' . esc_attr( $args['id'] ) . '" id="' . esc_attr( $args['id'] ) . '">';

			foreach( $args['options'] as $val => $name ) {
				$ui .= '<option value="' . esc_attr( $val ) . '" ' . selected( $val, $value, false ) . '>' . $name . '</option>' . "\n";
			}

			$ui .= '</select>' . "\n";

			$ui .= '</div></div>' . "\n";
			$ui .= '</div>' . "\n";

			if ( ! empty( $args['description'] ) ) {
				$ui .= '<div><small>' . stripslashes( $args['description'] ) . '</small></div>';
			}

			return $ui;
		}

		/**
		 * Select option
		 *
		 * @return string
		 */
		public function ui_hide_title_elementor( $post, $args = array() ) {
			$args = wp_parse_args( $args,
				array(
					'id' => 'elementor_hide_title',
					'label' => '',
					'description' => '',
					'default' => ''
				) );

			if ( empty( $args['id'] ) ) {
				return false;
			}

			$value = $args['default'];
			$settings = get_post_meta( $post->ID, '_elementor_page_settings', true );

			if ( ! $settings || null === $settings ) {
				$value = $args['default'];
			} else {
				if ( isset( $settings['hide_title'] ) && "yes" === $settings['hide_title'] ) {
					$value = "yes";
				}
			}

			$ui = '<div class="components-panel__row">' . "\n";	
			//$ui .= '<div class="components-base-control"><div class="components-base-control__field">';

			$ui .= '<input type="checkbox" name="' . esc_attr( $args['id'] ) . '" id="' . esc_attr( $args['id'] ) . '" ' . checked( $value, 'yes', false ) . '>';
			$ui .= '<label style="margin-right: auto; position: relative; top: -2px;" for="' . esc_attr( $args['id'] ) . '">' . $args['label'] . '</label>';

			//$ui .= '</div></div>' . "\n";
			$ui .= '</div>' . "\n";

			if ( ! empty( $args['description'] ) ) {
				$ui .= '<div><small>' . stripslashes( $args['description'] ) . '</small></div>';
			}

			return $ui;
		}
	}
}
