<?php
/**
 * The skin loader
 * Require the core plugin
 *
 * @package Brayn
 * @since 1.0
 */

// Accessing directly?.
if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( 'Brayn_Skin_Loader' ) ) {
	/**
	 * This class is in charge of displaying custom skin.
	 * Compile the scss into css code
	 *
	 * @since 1.0
	 */
	class Brayn_Skin_Loader {
		/**
		 * Custom skin status holder
		 *
		 * @var active
		 * @access public
		 */
		public $active = false;

		/**
		 * Path status holder
		 *
		 * @var path_ready
		 * @access public
		 */
		public $path_ready = false;

		/**
		 * Custom css path
		 *
		 * @var css_path
		 * @access public
		 */
		public $css_path = false;

		/**
		 * Custom css url
		 *
		 * @var css_path
		 * @access public
		 */
		public $css_url = false;

		/**
		 * Scss import path
		 *
		 * @var import_path
		 * @access private
		 */
		private $import_path = '';

		/**
		 * The constructor
		 *
		 * @since 1.0
		 * @return void
		 */
		public function __construct() {
			// Dont bother if user decide to turn off the core plugin.
			if ( ! class_exists( 'Brayn_Core' ) ) {
				return false;
			}

			$this->set_status();
			$this->setup_path();

			// Set the scss import path.
			$this->import_path = get_parent_theme_file_path( '/assets/scss/' );
		}

		/**
		 * Create path to place the css file if its not already exists
		 *
		 * @since 1.0
		 * @return void
		 */
		private function setup_path() {
			$upload_dir     = wp_upload_dir();
			$this->css_path = trailingslashit( $upload_dir['basedir'] ) . 'brayn-dynamic-css';
			$this->css_url  = trailingslashit( $upload_dir['baseurl'] ) . 'brayn-dynamic-css';

			if ( ! is_dir( $this->css_path ) ) {
				brayn_core_create_path( $this->css_path );

				// Checking once again, incase WP failed to create the dir.
				if ( is_dir( $this->css_path ) ) {
					$this->path_ready = true;
				}
			} else {
				$this->path_ready = true;
			}
		}

		/**
		 * Check if user has activate the custom skin or not
		 *
		 * @return bool
		 */
		public function set_status() {
			$status_check = get_theme_mod( 'enable_custom_skin' );
			
			if ( empty( $status_check ) ) {
				$status = false;
			} else {
				$status = true;
			}

			$this->active = $status;
			return $this->active;
		}

		/**
		 * get css url
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_css_url() {
			if ( ! $this->path_ready ) {
				return false;
			}

			if ( $this->css_url ) {
				return $this->css_url;
			}

			$upload_dir     = wp_upload_dir();
			$this->css_url  = trailingslashit( $upload_dir['baseurl'] ) . 'brayn-dynamic-css';
			return $this->css_url;
		}

		/**
		 * Write the compiled css code into a file
		 *
		 * @since 1.0
		 * @return bool
		 */
		public function write_css_to_file() {
			// Cancel if we failed to create the path.
			if ( ! $this->path_ready ) {
				set_theme_mod( 'brayn-failed-write-css', 'yes' );
				return false;
			}

			$css_code = $this->get_compiled_css();
			$css_file = $this->css_path . '/brayn-custom-skin.css';

			if ( brayn_core_write_to_file( $css_file, $css_code ) ) {
				remove_theme_mod( 'brayn-failed-write-css' );
				set_theme_mod( 'brayn-custom-css-version', intval( time() ) );
				return true;
			} else {
				set_theme_mod( 'brayn-failed-write-css', 'yes' );
				remove_theme_mod( 'brayn-custom-css-version' );
				return false;
			}

			return false;
		}

		/**
		 * Running after customisation saved.
		 *
		 * @since 1.0
		 * @return bool
		 */
		public function take_action() {
			if ( $this->set_status() ) {
				$this->save_all();
			} else {
				$this->save_typography();
			}

			$this->write_css_to_file();
		}

		/**
		 * Get full compiled css
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_compiled_css() {
			$precompiled_code = $this->get_precompiled_scss();
			$module = new Brayn_Core_Compiler_Scss( $precompiled_code, $this->import_path );
			$css_code = $module->compile();

			return $css_code;
		}

		/**
		 * Get typograhy compiled css
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_compiled_typography_css() {
			$precompiled_code = $this->get_precompiled_typography_scss();
			$module = new Brayn_Core_Compiler_Scss( $precompiled_code, $this->import_path );
			$css_code = $module->compile();

			return $css_code;
		}

		/**
		 * Get editor style compiled css
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_compiled_editor_style_css() {
			$precompiled_code = $this->get_precompiled_editor_style_scss();
			$module = new Brayn_Core_Compiler_Scss( $precompiled_code, $this->import_path );
			$css_code = $module->compile();

			return $css_code;
		}

		/**
		 * Save typography, running after customizer save
		 *
		 * @since 1.0
		 * @return string
		 */
		public function save_typography() {
			$css = $this->get_compiled_typography_css();
			set_theme_mod( 'brayn_typography_skin_code', $css );
		}

		/**
		 * Save all, running after customizer save
		 *
		 * @since 1.0
		 * @return string
		 */
		public function save_all() {
			$css = $this->get_compiled_css();
			set_theme_mod( 'brayn_custom_skin_code', $css );
		}

		/**
		 * Get precompiled scss based on user settings
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_precompiled_scss() {
			// Primary color.
			$primary = get_theme_mod( 'brayn_primary_color', '#46ad2a' );

			// Backgrounds.
			$body_bg          = get_theme_mod( 'body_bg', '#ffffff' );
			$heading_bg       = get_theme_mod( 'page_heading_bg', '#212121' );
			$preloader_bg     = get_theme_mod( 'preloader_bg', '#1a1a1a' );
			$preloader_loadbg = get_theme_mod( 'preloader_loadbg', '#46ad2a' );
			$footer_bg        = get_theme_mod( 'footer_bg', '#212121' );
			$input_bg         = get_theme_mod( 'input_bg', '#ffffff' );
			$input_bg_focus   = get_theme_mod( 'input_bg_focus', '#ffffff' );
			$menu_bg          = get_theme_mod( 'menu_bg', '#0a0a0a' );
			$searchbar_bg     = get_theme_mod( 'searchbar_bg', '#0a0a0a' );

			// Texts.
			$text_main              = get_theme_mod( 'color_text_main', '#565656' );
			$text_heading           = get_theme_mod( 'color_text_heading', '#1a1a1a' );
			$text_page_heading      = get_theme_mod( 'color_text_page_heading', '#f4f4f4' );
			$text_page_heading_text = get_theme_mod( 'color_text_page_heading_text', '#afafaf' );
			$text_footer            = get_theme_mod( 'color_text_footer', '#dadada' );
			$text_footer_heading    = get_theme_mod( 'color_text_footer_heading', '#f4f4f4' );
			$text_input             = get_theme_mod( 'color_text_input', '#989898' );
			$text_input_focus       = get_theme_mod( 'color_text_input_focus', '#565656' );
			$text_input_placeholder = get_theme_mod( 'color_text_input_placeholder', '#989898' );

			// Links.
			$color_link              = get_theme_mod( 'color_link', '#46ad2a' );
			$color_link_hover        = get_theme_mod( 'color_link_hover', '#255b16' );
			$color_link_menu         = get_theme_mod( 'color_link_menu', '#afafaf' );
			$color_link_menu_hover   = get_theme_mod( 'color_link_menu_hover', '#ffffff' );
			$color_link_menu_active  = get_theme_mod( 'color_link_menu_active', '#ffffff' );
			$color_link_footer       = get_theme_mod( 'color_link_footer', '#dadada' );
			$color_link_footer_hover = get_theme_mod( 'color_link_footer_hover', '#ffffff' );

			// If custom skin disabled, we should role back the skin to the default one!.
			if ( ! is_customize_preview() && ! $this->set_status() ) {
				// Primary color.
				$primary = '#46ad2a';

				// Backgrounds.
				$body_bg          = '#ffffff';
				$heading_bg       = '#212121';
				$preloader_bg     = '#1a1a1a';
				$preloader_loadbg = '#46ad2a';
				$footer_bg        = '#212121';
				$input_bg         = '#ffffff';
				$input_bg_focus   = '#ffffff';
				$menu_bg          = '#0a0a0a';
				$searchbar_bg     = '#0a0a0a';

				// Texts.
				$text_main              = '#565656';
				$text_heading           = '#1a1a1a';
				$text_page_heading      = '#f4f4f4';
				$text_page_heading_text = '#afafaf';
				$text_footer            = '#dadada';
				$text_footer_heading    = '#f4f4f4';
				$text_input             = '#989898';
				$text_input_focus       = '#565656';
				$text_input_placeholder = '#989898';

				// Links.
				$color_link              = '#46ad2a';
				$color_link_hover        = '#255b16';
				$color_link_menu         = '#afafaf';
				$color_link_menu_hover   = '#ffffff';
				$color_link_menu_active  = '#ffffff';
				$color_link_footer       = '#dadada';
				$color_link_footer_hover = '#ffffff';			
			}

			// font sizes.
			$base_font_size = get_theme_mod( 'body_font_size', '17' );
			$font_size_h1   = get_theme_mod( 'heading_h1', '2.75' );
			$font_size_h2   = get_theme_mod( 'heading_h2', '2' );
			$font_size_h3   = get_theme_mod( 'heading_h3', '1.75' );
			$font_size_h4   = get_theme_mod( 'heading_h4', '1.5' );
			$font_size_h5   = get_theme_mod( 'heading_h5', '1.25' );
			$font_size_h6   = get_theme_mod( 'heading_h6', '1' );
			$line_height    = get_theme_mod( 'body_line_height', '1.7' );

			// fonts.
			$heading_font = get_theme_mod( 'brayn_heading_font', WIP_THEMES_DEFAULT_HEADING_FONT );
			$body_font    = get_theme_mod( 'brayn_body_font', WIP_THEMES_DEFAULT_BODY_FONT );
			
			// heading font name.
			$heading_font_name = explode( ':', $heading_font );
			$heading_font_name = str_replace( '+', ' ', $heading_font_name[0] );

			// Body font name.
			$body_font_name = explode( ':', $body_font );
			$body_font_name = str_replace( '+', ' ', $body_font_name[0] );

			$scss = '
$grid-breakpoints: (
  xs: 0,
  smm: 440px,
  sm: 576px,
  md: 768px,
  lg: 992px,
  xl: 1200px,
  wd: 1310px
)!default;

@braynwp_scss_import "mixins";

// primary the accent color
$color__primary: ' . $primary . ' !default; 

// backgrounds
$color__body-background: ' . $body_bg . ' !default;
$color__light-background: set-diff-color($color__body-background, 2.5%);
$color__page-heading-background: ' . $heading_bg . ' !default;
$color__preloader-background: ' . $preloader_bg . ' !default;
$color__preloader-loadbg: ' . $preloader_loadbg . ' !default;
$color__footer-background: ' . $footer_bg . ' !default;
$color__input-background: ' . $input_bg . '	!default;
$color__input-background-focus: ' . $input_bg_focus . '	!default;
$color__menu-background: ' . $menu_bg . ' !default;
$color__searchbar-background: ' . $searchbar_bg . ' !default;

// Texts
$color__text-main: ' . $text_main . ';
$color__text-heading: ' . $text_heading . ';
$color__page-heading: ' . $text_page_heading . ';
$color__page-heading-text: ' . $text_page_heading_text . ';
$color__text-soft: lighten($color__text-main, 34%);
$color__text-footer-heading: ' . $text_footer_heading . ';
$color__text-footer: ' . $text_footer . ';
$color__text-input: ' . $text_input . ';
$color__text-input-focus: ' . $text_input_focus . ';
$color__text-input-placeholder: ' . $text_input_placeholder . ';

// Links
$color__link: ' . $color_link . ' !default;
$color__link-hover: ' . $color_link_hover . ' !default;
$color__link-menu: ' . $color_link_menu . ' !default;
$color__link-menu-hover: ' . $color_link_menu_hover . ' !default;
$color__link-menu-active: ' . $color_link_menu_active . ' !default;
$color__link-footer: ' . $color_link_footer . ' !default;
$color__link-footer-hover: ' . $color_link_footer_hover . ' !default;

// Borders (auto compile)
$color__content-border: set-diff-color($color__body-background, 9%);
$color__form-border: set-diff-color($color__body-background, 9%);
$color__footer-border: set-diff-color($color__footer-background, 9%);
$color__menu-border: set-diff-color($color__menu-background, 7.5%);

// font sizes
$size__font-base: ' . $base_font_size . ' !default;
$size__font-h1: ' . $font_size_h1 . 'rem !default;
$size__font-h2: ' . $font_size_h2 . 'rem !default;
$size__font-h3: ' . $font_size_h3 . 'rem !default;
$size__font-h4: ' . $font_size_h4 . 'rem !default;
$size__font-h5: ' . $font_size_h5 . 'rem !default;
$size__font-h6: ' . $font_size_h6 . 'rem !default;

$line-height: ' . $line_height . ' !default;
$base-font-weight: 400 !default;

$font-family-sans-serif: 	"' . $body_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;
$font-family-monospace: 	SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace !default;
$font-family-base: 			$font-family-sans-serif !default;
$font-family-heading:		"' . $heading_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;

// @version 2.0.0 - woocommerce
$woo__onsale-bg: ' . get_theme_mod( 'woo_onsale_bg', '#e74c3c' ) . ' !default;
$woo__onsale-text: ' . get_theme_mod( 'woo_onsale_text', '#ffffff' ) . ' !default;
$woo__button-bg: ' . get_theme_mod( 'woo_button_bg', '#212121' ) . ' !default;
$woo__button-text: ' . get_theme_mod( 'woo_button_text', '#dadada' ) . ' !default;
$woo__button-bg-hover: ' . get_theme_mod( 'woo_button_bg_hover', '#46ad2a' ) . ' !default;
$woo__button-text-hover: ' . get_theme_mod( 'woo_button_text_hover', '#ffffff' ) . ' !default;
$woo__single-price-color: ' . get_theme_mod( 'woo_single_price_color', '#46ad2a' ) . ' !default;
$woo__store-notice-bg: ' . get_theme_mod( 'woo_store_notice_bg', '#212121' ) . ' !default;
$woo__store-notice-color: ' . get_theme_mod( 'woo_store_notice_color', '#dadada' ) . ' !default;
$woo__store-notice-link: ' . get_theme_mod( 'woo_store_notice_link', '#ffffff' ) . ' !default;
$woo__top-cart-count-bg: ' . get_theme_mod( 'woo_topcart_count_bg', '#e3251d' ) . ' !default;
$woo__top-cart-count-color: ' . get_theme_mod( 'woo_topcart_count_text', '#ffffff' ) . ' !default;

@braynwp_scss_import "skin-code";
';

			return str_replace( 'braynwp_scss_import', 'import', $scss );

		}

		/**
		 * Get precompiled scss for font settings and typography
		 * Some users might just want to change the font settings but keep the default theme skin
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_precompiled_typography_scss() {
			// font sizes.
			$base_font_size = get_theme_mod( 'body_font_size', '17' );
			$font_size_h1   = get_theme_mod( 'heading_h1', '2.75' );
			$font_size_h2   = get_theme_mod( 'heading_h2', '2' );
			$font_size_h3   = get_theme_mod( 'heading_h3', '1.75' );
			$font_size_h4   = get_theme_mod( 'heading_h4', '1.5' );
			$font_size_h5   = get_theme_mod( 'heading_h5', '1.25' );
			$font_size_h6   = get_theme_mod( 'heading_h6', '1' );
			$line_height    = get_theme_mod( 'body_line_height', '1.7' );

			// fonts.
			$heading_font = get_theme_mod( 'brayn_heading_font', WIP_THEMES_DEFAULT_HEADING_FONT );
			$body_font    = get_theme_mod( 'brayn_body_font', WIP_THEMES_DEFAULT_BODY_FONT );
			
			// heading font name.
			$heading_font_name = explode( ':', $heading_font );
			$heading_font_name = str_replace( '+', ' ', $heading_font_name[0] );

			// Body font name.
			$body_font_name = explode( ':', $body_font );
			$body_font_name = str_replace( '+', ' ', $body_font_name[0] );

			$scss = '
$grid-breakpoints: (
  xs: 0,
  smm: 440px,
  sm: 576px,
  md: 768px,
  lg: 992px,
  xl: 1200px,
  wd: 1310px
)!default;

@import "mixins";

// font sizes
$size__font-base: ' . $base_font_size . ' !default;
$size__font-h1: ' . $font_size_h1 . 'rem !default;
$size__font-h2: ' . $font_size_h2 . 'rem !default;
$size__font-h3: ' . $font_size_h3 . 'rem !default;
$size__font-h4: ' . $font_size_h4 . 'rem !default;
$size__font-h5: ' . $font_size_h5 . 'rem !default;
$size__font-h6: ' . $font_size_h6 . 'rem !default;

$line-height: ' . $line_height . ' !default;
$base-font-weight: 400 !default;

$font-family-sans-serif: 	"' . $body_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;
$font-family-monospace: 	SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace !default;
$font-family-base: 			$font-family-sans-serif !default;
$font-family-heading:		"' . $heading_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;

:root {
	font-size: calculation-fontsize($size__font-base, 0.85);
}
@brayn_scss_include media-breakpoint-up(sm) {
	:root {
		font-size: calculation-fontsize($size__font-base, 0.9);
	}
}
@brayn_scss_include media-breakpoint-up(md) {
	:root {
		font-size: calculation-fontsize($size__font-base, 0.95);
	}
}
@brayn_scss_include media-breakpoint-up(lg) {
	:root {
		font-size: $size__font-base+(px);
	}
}

html {
	font-size: 1em;
}

body {
	font-family: $font-family-base;
	font-size: 1rem;
	font-weight: $base-font-weight;
	line-height: $line-height;
}
h1,h2,h3,h4,h5,h6 {
	font-family: $font-family-heading;
}

h1 {
	font-size: $size__font-h1;
	line-height: 1.4;
}
h2 {
	font-size: $size__font-h2;
	line-height: 1.4;
}
h3 {
	font-size: $size__font-h3;
	line-height: 1.4;
}
h4 {
	font-size: $size__font-h4;
	line-height: 1.4;
}
h5 {
	font-size: $size__font-h5;
	line-height: 1.4;
}
h6 {
	font-size: $size__font-h6;
}

pre,
code,
kbd,
samp {
	font-family: $font-family-monospace;
}
.main-menu-container-wrap {
	ul#brayn-menu {
		font-family: $font-family-heading;
	}
}
.br-animated-number-wrap {
	.br-animated-number {
		font-family: $font-family-heading;
	}
}
.navigation {
	&.post-navigation {
		.nav-links {
			.nav-previous,
			.nav-next {
				a {
					.nav-title {
						font-family: $font-family-heading;
					}
				}
			}
		}
	}
}
.comments-area {
	#comments {
		#cancel-comment-reply-link {
			font-family: $font-family-base;
		}
	}
}';

			return str_replace( 'brayn_scss_include', 'include', $scss );
		}

		/**
		 * Get precompiled scss for editor style
		 *
		 * @since 1.0
		 * @return string
		 */
		public function get_precompiled_editor_style_scss() {
			// Require colors.
			$primary           = get_theme_mod( 'brayn_primary_color', '#46ad2a' );
			$body_bg           = get_theme_mod( 'body_bg', '#ffffff' );
			$text_main         = get_theme_mod( 'color_text_main', '#565656' );
			$text_heading      = get_theme_mod( 'color_text_heading', '#1a1a1a' );
			$color_link        = get_theme_mod( 'color_link', '#ea3c53' );
			$color_link_hover  = get_theme_mod( 'color_link_hover', 'darken($color__link, 20%)' );
			
			$base_font_size    = get_theme_mod( 'body_font_size', '17' );
			$font_size_h1      = get_theme_mod( 'heading_h1', '2.75' );
			$font_size_h2      = get_theme_mod( 'heading_h2', '2' );
			$font_size_h3      = get_theme_mod( 'heading_h3', '1.75' );
			$font_size_h4      = get_theme_mod( 'heading_h4', '1.5' );
			$font_size_h5      = get_theme_mod( 'heading_h5', '1.25' );
			$font_size_h6      = get_theme_mod( 'heading_h6', '1' );
			$line_height       = get_theme_mod( 'body_line_height', '1.7' );

			// Some need to set in default state when custom skin is not active.
			if ( ! $this->set_status() ) {
				$primary           = '#46ad2a';
				$body_bg           = '#ffffff';
				$text_main         = '#565656';
				$text_heading      = '#1a1a1a';
				$color_link        = '#46ad2a';
				$color_link_hover  = 'darken($color__link, 20%)';	
			}

			// Fonts.
			$heading_font = get_theme_mod( 'brayn_heading_font', WIP_THEMES_DEFAULT_HEADING_FONT );
			$body_font    = get_theme_mod( 'brayn_body_font', WIP_THEMES_DEFAULT_BODY_FONT );
			
			// Heading font name.
			$heading_font_name = explode( ':', $heading_font );
			$heading_font_name = str_replace( '+', ' ', $heading_font_name[0] );

			// Body font name.
			$body_font_name = explode( ':', $body_font );
			$body_font_name = str_replace( '+', ' ', $body_font_name[0] );

			$scss = '
$grid-breakpoints: (
  xs: 0,
  smm: 440px,
  sm: 576px,
  md: 768px,
  lg: 992px,
  xl: 1200px,
  wd: 1310px
)!default;

@import "mixins";

$color__primary: ' . $primary . ' !default;
$color__body-background: ' . $body_bg . '	!default;
$color__text-main: ' . $text_main . ';
$color__text-heading: ' . $text_heading . ';
$color__text-soft: lighten($color__text-main, 34%);
$color__link: ' . $color_link . ' !default;
$color__link-hover: ' . $color_link_hover . ' !default;

$color__content-border: set-diff-color($color__body-background, 9%);

$size__font-base: ' . $base_font_size . ' !default;
$size__font-h1: ' . $font_size_h1 . 'rem !default;
$size__font-h2: ' . $font_size_h2 . 'rem !default;
$size__font-h3: ' . $font_size_h3 . 'rem !default;
$size__font-h4: ' . $font_size_h4 . 'rem !default;
$size__font-h5: ' . $font_size_h5 . 'rem !default;
$size__font-h6: ' . $font_size_h6 . 'rem !default;

$line-height: ' . $line_height . '!default;
$base-font-weight: 400 !default;

$font-family-sans-serif: 	"' . $body_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;
$font-family-monospace: 	SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace !default;
$font-family-base: 			$font-family-sans-serif !default;
$font-family-heading:		"' . $heading_font_name . '", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol", "Noto Color Emoji" !default;

body {
  font-family: $font-family-base;
  color: $color__text-main;
  background-color: $color__body-background;
  line-height: $line-height;
  font-size: $size__font-base+"px";
}
h1, h2, h3, h4, h5, h6 {
  font-family: $font-family-heading;
  color: $color__text-heading;

 	a {
		color: $color__text-heading;

		&:hover {
			color: $color__link;
		}
	}
}

h1 {
	font-size: $size__font-h1;
	line-height: 1.35;
}
h2 {
	font-size: $size__font-h2;
	line-height: 1.15;
}
h3 {
	font-size: $size__font-h3;
	line-height: 1.15;
}
h4 {
	font-size: $size__font-h4;
	line-height: 1.15;
}
h5 {
	font-size: $size__font-h5;
	line-height: 1.15;
}
h6 {
	font-size: $size__font-h6;
}

a {
	color: $color__link;
	text-decoration: none;

	&:hover {
		color: $color__link-hover;
	}
}

pre {
	font-family: $font-family-monospace;
}

blockquote {
	color: $color__text-heading;
}

/* Main column width */
.wp-block {
    max-width: 710px;
}

/* Width of "wide" blocks */
.wp-block[data-align="wide"] {
    max-width: 75%;
}

/* Width of "full-wide" blocks */
.wp-block[data-align="full"] {
    max-width: none;
}

p,ul,ol {
	margin-top: 29px;
}
h1,
h2,
h3,
h4,
h5,
h6,
.wp-block-cover,
.wp-block-embed,
.wp-block-image:not(.alignleft):not(.alignright)  {
	margin-top: 50px;
	margin-bottom: 0;
	font-weight: 500;
}
.wp-block-cover,
.wp-block-embed,
.wp-block-image:not(.alignleft):not(.alignright)  {
	margin-bottom: 20px;
}
p.wp-block-cover-text {
	margin-top: 0;
}
h4 + p,
h3 + p,
h2 + p,
h1 + p {
	margin-top: 1rem;
}
h4 + *:not(p),
h3 + *:not(p),
h2 + *:not(p),
h1 + *:not(p) {
	margin-top: 1.5rem;
}
figure {
	margin-left: 0;
	margin-right: 0;

	&.alignleft {
		margin-top: 29px;
		margin-right: 2rem;
	}

	&.alignright {
		margin-top: 29px;
		margin-left: 2rem;
	}
}

ul,ol {
	list-style-position: outside;
	margin-left: 2rem;
	ul,ol {
		padding-left: 2rem;
		margin-top: .5rem;
		margin-bottom: 1rem;
		margin-left: 0;
	}
}
p.has-drop-cap:not(:focus)::first-letter {
	float: left;
	font-size: 3.7em;
	line-height: 0.68;
	font-weight: bold;
	margin: 0.4rem .85rem 0 0;
	text-transform: uppercase;
	font-style: normal;
}

.gallery-caption {
    margin: 0;
	padding: 1rem;
	background: rgba(255, 255, 255, 0.85);
	color: rgba(0, 0, 0, 0.75);
	overflow: initial;
}

table {
	display: table;
	width: 100%;
	border-collapse: collapse;
	border-spacing: 0;
	
	td, th {
		padding: .5rem;
		vertical-align: top;
	}

	thead {
		th {
			font-weight: 600;
		}
	}
}

.wp-block-table table,
table {
	border: 1px solid $color__content-border;

	thead th,
	tfoot td,
	tbody td {
		border: 1px solid $color__content-border;
	}

	td, th {
		padding: .5rem .5rem .5rem .5rem;
		vertical-align: top;
	}

	thead {
		th {
			font-weight: 600;
		}
	}

	.wp-block-table__cell-content {
		padding: 0;
	}

	thead th,
	tfoot td {
		background-color: set-diff-color( $color__body-background, 2%);
	}

	&.has-background {
		thead th,
		tfoot td {
			background-color: rgba(0,0,0, .05);
		}			
	}

	&.is-style-stripes {
		border: 1px solid $color__content-border;
		border-bottom: 1px solid set-diff-color( $color__content-border, 5% );
		border-right: 1px solid set-diff-color( $color__content-border, 5% );
		
		thead th {
			border: 0;
			background-color: set-diff-color( $color__body-background, 5%);
		}
		tr:nth-child(odd) {
			background-color: set-diff-color( $color__body-background, 2%);
		}
	}
}
pre {
	&,
	&.wp-block-preformatted,
	&.wp-block-code {
		margin-top: 50px;
		padding: 2rem;
		background: black;
		color: white;
		border-radius: 2px;
		tab-size: 4;

		+ * {
			margin-top: 50px;
		}
	}
}
.wp-block-code {
	margin-top: 50px;
	padding: 2rem;
	background: black;
	color: white;
	border-radius: 2px;
	tab-size: 4;

	textarea {
		background: black;
		border: 0;
		color: white;
	}

	+ * {
		margin-top: 50px;
	}
}
p code {
	padding: .25rem .25rem;
	font-size: .75rem;
	line-height: 1;
	background: set-diff-color( $color__body-background, 3%);
	border: 1px solid set-diff-color( $color__body-background, 7%);
	color: inherit;
	border-radius: 3px;
}


blockquote {
	&,
	&.wp-block-quote{
		position: relative;
		padding-left: 2rem;
		padding-right: 2rem;
		padding-top: 4rem !important;
		margin-top: 50px !important;
		text-align: center;
		min-height: 51px;
		border-left-width: 0;

		&:not(.is-large),
		&:not(.is-style-large) {
			border-left-width: 0;
		}
	}
	

	p {
		font-family: "Georgia", serif;
		font-size: 1.25rem;
	}

	&:before {
		content: "";
		position: absolute;
		left: 50%;
		top: 0;
		width: 3rem;
		height: 3rem;
		line-height: 3rem;
		text-align: center;
		margin: 0 0 0 -1.5rem;
		background-image: url(../images/quote.svg);
		background-position: center;
		background-repeat: no-repeat;
		background-size: 1.5rem;
		background-color: #ececec;
		border-radius: 50%;
	}

	cite {
		font-size: .75rem !important;
		font-weight: 600;
		position: relative;
		font-style: normal;
	}

	+ * {
		margin-top: 50px;
	}
}

ul.wp-block-latest-posts {
	margin-top: 50px;
	margin-bottom: 50px;
	margin-left: 0;
	padding-left: 0;
	padding-right: 0;
	list-style-type: none;

	li {
		list-style-type: none;
		margin: 0;
		padding: 1rem;

		time {
			padding: .5rem 0 .25rem;
			font-size: 11px;
			line-height: 1;
			font-weight: 500;
			text-transform: uppercase;
			letter-spacing: .1em;			
		}
	}

	&.is-grid {
		margin-left: -8px;
		margin-right: -8px;
		li {
			position: relative;
			margin: 0 8px 1rem 8px;
			padding: 0 0 1rem 0;

			&:after {
				content: "";
				position: absolute;
				left: 0;
				right: 0;
				bottom: 0;
				height: 1px;
			}
		}
	}

	+ * {
		margin-top: 0;
	}
}

.wp-block-categories {
	.wp-block-categories__list {
		list-style-type: none;
		margin: 50px 0 50px 0;
		padding: 0;

		li {
			list-style-type: none;
			position: relative;
			padding: .5rem 0 .5rem 32px;
			margin-left: 0;
			font-size: 12px;
			text-align: left;
			border-bottom: 1px solid $color__content-border;

			a {
				font-size: 1rem;
				display: inline-block;
				margin-right: .5rem;
			}

			&:before {
				content: "\f0ea";
				display: inline-block;
				font: normal normal normal 14px/1 FontAwesome;
				font-size: 13px;
				text-rendering: auto;
				-webkit-font-smoothing: antialiased;
				-moz-osx-font-smoothing: grayscale;
				position: absolute;
				left: 0;
				width: 20px;
				top: .9rem;
				text-align: center;
				opacity: .75;
			}

			ul {
				margin-left: -32px;
				margin-top: .5rem;
				margin-bottom: 0;
				padding-left: 32px;
				border-top: 1px solid $color__content-border;

				li:last-child {
					border-bottom-width: 0;
				}
			}
		}


		+ * {
			margin-top: 0;
		}
	}
}

ul.wp-block-archives {
	&.wp-block-archives-list {
		list-style-type: none;
		margin: 50px 0 50px;
		padding: 0;

		li {
			list-style-type: none;
			position: relative;
			padding: .5rem 0 .5rem 32px;
			margin-left: 0;
			font-size: 12px;

			a {
				font-size: 1rem;
				display: inline-block;
				margin-right: .5rem;
			}

			&:before {
				content: "\f133";
				display: inline-block;
				font: normal normal normal 14px/1 FontAwesome;
				font-size: 13px;
				text-rendering: auto;
				-webkit-font-smoothing: antialiased;
				-moz-osx-font-smoothing: grayscale;
				position: absolute;
				left: 0;
				width: 20px;
				top: .9rem;
				text-align: center;
				opacity: .75;
			}

			ul {
				margin-left: -32px;
				margin-top: .5rem;
				padding-left: 32px;

				li:last-child {
					border-bottom-width: 0;
				}
			}
		}


		+ * {
			margin-top: 0;
		}
	}
}

ul.wp-block-latest-posts {
	li {
		border: 1px solid $color__content-border;
		border-bottom-width: 0;

		&:last-child {
			border-bottom-width: 1px;
		}
	}
	&.is-grid {
		li {
			border: 0;
			
			&:after {
				background: $color__content-border;
			}
		}
	}

	&:not(.is-grid) {
		li {
			-webkit-transition: background .3s ease-out 0s;
			transition: background .3s ease-out 0s;

			&:hover {
				background: set-diff-color( $color__body-background, 1%);
			}
		}
	}
}

ul.wp-block-categories {
	&.wp-block-categories-list {
		li {
			border-bottom: 1px solid $color__content-border;

			ul {
				border-top: 1px solid $color__content-border;
			}
		}
	}
}

ul.wp-block-archives {
	&.wp-block-archives-list {
		li {
			border-bottom: 1px solid $color__content-border;

			ul {
				border-top: 1px solid $color__content-border;
			}
		}
	}
}

ol.wp-block-latest-comments {
	list-style-type: none;
	margin-left: 0 !important;
	padding-left: 0 !important;
	margin-top: 50px;
	margin-bottom: 50px;
	.wp-block-latest-comments__comment {
		border-bottom: 1px solid $color__content-border;
		margin-bottom: 1rem;
		padding-bottom: 1rem;
	}
	.wp-block-latest-comments__comment-author {
		display: inline-block;
		padding: 0 .35rem 0 .35rem;
		margin-right: .25rem;
		border-radius: 3px;
		font-size: .675rem;
		height: 20px;
		line-height: 20px;
		vertical-align: baseline;
		text-transform: uppercase;
		letter-spacing: .15em;
		border: 1px solid $color__link;
		color: $color__link;
	}
	.wp-block-latest-comments__comment-link {
		font-weight: 600;
	}
	.wp-block-latest-comments__comment-meta time {
		font-size: .675rem;
		text-transform: uppercase;
		letter-spacing: .15em;
		margin-top: .5rem;
	}
	.wp-block-latest-comments__comment-excerpt {
		padding: 1rem;
		border-radius: 3px;
		margin-top: 1rem;
		background: set-diff-color($color__body-background, 3%);

		p {
			margin: 0 0 0;
		}
	}
}

.wp-block-tag-cloud {
	margin: 50px 0 50px;
	padding: 0;
	a {
		display: inline-block;
		font-size: .75rem !important;
		text-transform: uppercase;
		letter-spacing: .025em;
		line-height: 1;
		font-weight: 600;
		padding: .25rem .3rem;
		border-radius: 2px;
		vertical-align: middle;
		background: $color__link;
		color: set-diff-color($color__link, 70%) !important;

		&:hover {
			background: set-diff-color($color__link, 5%);
		}
	}
}

.wp-block-file {
	padding: 1rem;
	margin-top: 29px;
	position: relative;
	display: -ms-flexbox;
	display: flex;
	-ms-flex-wrap: wrap;
	flex-wrap: wrap;
	-ms-flex-pack: justify;
	justify-content: space-between;
	-ms-align-items: center;
	align-items: center;
	border-radius: 3px;
	background: rgba(0,0,0, .05);

	.wp-block-file__button {
		border-radius: 2px;
		text-transform: uppercase;
	}
}

ul.wp-block-gallery {
	margin-top: 50px;
}

.wp-block-pullquote {
	margin-top: 50px;

	blockquote {
		margin-top: 0 !important;
	}

	+ * {
		margin-top: 50px;
	}

	&.alignleft,
	&.alignright {
		margin-top: 20px;
		margin-bottom: 2rem;

		+ * {
			margin-top: 29px;
		}
	}

	&.is-style-solid-color {
		blockquote {
			max-width: 80%;
		}
	}
}

.wp-block-button {
	margin-top: 29px;
	margin-bottom: 15px;
	.wp-block-button__link {
		font-size: 1rem;
		line-height: 1;
		padding: 1rem 1.5rem;
	}
	&.is-style-squared {
		.wp-block-button__link {
			-webkit-border-radius: 2px;
			border-radius: 2px;
		}
	}
	&.is-style-default,
	&.is-style-squared {
		.wp-block-button__link {
			box-shadow: 0 3px 15px -1px rgba(0,0,0, .12);
			
			&:hover {
				box-shadow: inset 0 0 200px rgba(255,255,255, .1), 0 1px 2px 0 rgba(0,0,0, .3);
			}
		}
	}
	&.is-style-outline {
		.wp-block-button__link:hover {
			opacity: .65;
		}
	}
}

.wp-block-audio {
	margin-top: 29px;
	.mejs-container {
		width: 100% !important;
	}
}

hr.wp-block-separator {
	clear: both;
	text-align: center;
	margin: 50px auto 50px;

	&:not(.is-style-wide):not(.is-style-dots) {
		margin-left: auto;
		margin-right: auto;
	}

	+ * {
		margin-top: 0;
	}
}

.wp-block-embed__wrapper {
	iframe {
		width: 100%;
		max-width: 100% !important;
	}
}

.wp-block-cover,
.wp-block-cover-image {
	min-height: 50vh;
}

hr.wp-block-separator {
	border: 0;
	border-bottom: 2px solid $color__text-main;

	&.is-style-wide {
		border-bottom: 1px solid $color__content-border;
	}
	&.is-style-dots {
		border-bottom-width: 0;

		&:before {
			color: $color__text-main;
		}
	}
}

.mce-content-body {
	font-family: $font-family-base;
	color: $color__text-main;
	background-color: $color__body-background;
	line-height: $line-height;
	font-size: $size__font-base+"px";

	h1, h2, h3, h4, h5, h6 {
	  font-family: $font-family-heading;
	  color: $color__text-heading;

	 	a {
			color: $color__text-heading;

			&:hover {
				color: $color__link;
			}
		}
	}

	h1 {
		font-size: $size__font-h1;
		line-height: 1.35;
	}
	h2 {
		font-size: $size__font-h2;
		line-height: 1.15;
	}
	h3 {
		font-size: $size__font-h3;
		line-height: 1.15;
	}
	h4 {
		font-size: $size__font-h4;
		line-height: 1.15;
	}
	h5 {
		font-size: $size__font-h5;
		line-height: 1.15;
	}
	h6 {
		font-size: $size__font-h6;
	}

	a {
		color: $color__link;
		text-decoration: none;

		&:hover {
			color: $color__link-hover;
		}
	}

	pre {
		font-family: $font-family-monospace;
	}

	blockquote {
		color: $color__text-heading;
	}

	p,ul,ol {
		margin-top: 29px;
	}
	h1,
	h2,
	h3,
	h4,
	h5,
	h6,
	.wp-block-cover,
	.wp-block-embed,
	.wp-block-image:not(.alignleft):not(.alignright)  {
		margin-top: 50px;
		margin-bottom: 0;
		font-weight: 500;
	}
	.wp-block-cover,
	.wp-block-embed,
	.wp-block-image:not(.alignleft):not(.alignright)  {
		margin-bottom: 20px;
	}
	p.wp-block-cover-text {
		margin-top: 0;
	}
	h4 + p,
	h3 + p,
	h2 + p,
	h1 + p {
		margin-top: 1rem;
	}
	h4 + *:not(p),
	h3 + *:not(p),
	h2 + *:not(p),
	h1 + *:not(p) {
		margin-top: 1.5rem;
	}
	figure {
		margin-left: 0;
		margin-right: 0;

		&.alignleft {
			margin-top: 29px;
			margin-right: 2rem;
		}

		&.alignright {
			margin-top: 29px;
			margin-left: 2rem;
		}
	}

	ul,ol {
		list-style-position: outside;
		margin-left: 2rem;
		ul,ol {
			padding-left: 2rem;
			margin-top: .5rem;
			margin-bottom: 1rem;
			margin-left: 0;
		}
	}
	p.has-drop-cap:not(:focus)::first-letter {
		float: left;
		font-size: 3.7em;
		line-height: 0.68;
		font-weight: bold;
		margin: 0.4rem .85rem 0 0;
		text-transform: uppercase;
		font-style: normal;
	}

	.gallery-caption {
	    margin: 0;
		padding: 1rem;
		background: rgba(255, 255, 255, 0.85);
		color: rgba(0, 0, 0, 0.75);
		overflow: initial;
	}

	.wp-block-table table,
	table {
		width: 100%;
		border-collapse: collapse;
		border: 1px solid $color__content-border;

		thead th,
		tfoot td,
		tbody td {
			border: 1px solid $color__content-border;
		}

		thead th,
		tfoot td {
			background-color: set-diff-color( $color__body-background, 2%);
		}

		&.has-background {
			thead th,
			tfoot td {
				background-color: rgba(0,0,0, .05);
			}			
		}

		&.is-style-stripes {
			border: 1px solid $color__content-border;
			border-bottom: 1px solid set-diff-color( $color__content-border, 5% );
			border-right: 1px solid set-diff-color( $color__content-border, 5% );
			
			thead th {
				border: 0;
				background-color: set-diff-color( $color__body-background, 5%);
			}
			tr:nth-child(odd) {
				background-color: set-diff-color( $color__body-background, 2%);
			}
		}
	}
	pre {
		&,
		&.wp-block-preformatted,
		&.wp-block-code {
			margin-top: 50px;
			padding: 2rem;
			background: black;
			color: white;
			border-radius: 2px;
			tab-size: 4;

			+ * {
				margin-top: 50px;
			}
		}
	}
	.wp-block-code {
		margin-top: 50px;
		padding: 2rem;
		background: black;
		color: white;
		border-radius: 2px;
		tab-size: 4;

		textarea {
			background: black;
			border: 0;
			color: white;
		}

		+ * {
			margin-top: 50px;
		}
	}
	p code {
		padding: .25rem .25rem;
		font-size: .75rem;
		line-height: 1;
		background: set-diff-color( $color__body-background, 3%);
		border: 1px solid set-diff-color( $color__body-background, 7%);
		color: inherit;
		border-radius: 3px;
	}


	blockquote {
		&,
		&.wp-block-quote{
			position: relative;
			padding-left: 2rem;
			padding-right: 2rem;
			padding-top: 4rem !important;
			margin-top: 50px !important;
			text-align: center;
			min-height: 51px;
			border-left-width: 0;

			&:not(.is-large),
			&:not(.is-style-large) {
				border-left-width: 0;
			}
		}
		

		p {
			font-family: "Georgia", serif;
			font-size: 1.25rem;
		}

		&:before {
			content: "";
			position: absolute;
			left: 50%;
			top: 0;
			width: 3rem;
			height: 3rem;
			line-height: 3rem;
			text-align: center;
			margin: 0 0 0 -1.5rem;
			background-image: url(../images/quote.svg);
			background-position: center;
			background-repeat: no-repeat;
			background-size: 1.5rem;
			background-color: #ececec;
			border-radius: 50%;
		}

		cite {
			font-size: .75rem !important;
			font-weight: 600;
			position: relative;
			font-style: normal;
		}

		+ * {
			margin-top: 50px;
		}
	}
}';

			return str_replace( 'brayn_scss_include', 'include', $scss );
		}
	}
}
