<?php
/**
 * custom elementor metro gallery widget
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 1.0
 */

/**
 * Elementor Metro Gallery Widget.
 *
 * @since 1.0.0
 */
class Brayn_Elementor_Metro_Gallery_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-metro-gallery';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Metro Gallery', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-gallery-justified';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'images',
			[
				'label' => __( 'Add Images', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::GALLERY,
				'default' => [],
				'show_label' => false,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'gallery_link',
			[
				'label' => __( 'Link', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'file',
				'options' => [
					'file' => __( 'Media File', 'brayn-core' ),
					'none' => __( 'None', 'brayn-core' ),
				],
			]
		);

		$this->add_control(
			'open_lightbox',
			[
				'label' => __( 'Lightbox', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => __( 'Default', 'brayn-core' ),
					'yes' => __( 'Yes', 'brayn-core' ),
					'no' => __( 'No', 'brayn-core' ),
				],
				'condition' => [
					'gallery_link' => 'file',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings    = $this->get_settings_for_display();
		
		if ( empty( $settings['images'] ) ) {
			return;
		}

		$galleries = [];
		$total = count( $settings['images'] );
		$settings['thumbnail_size'] = 'large';
		$settings['thumbnail_custom_dimension'] = '';
		
		$c = 0;
		foreach ( $settings['images'] as $index => $attachment ) {
			$c++;
			$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src( $attachment['id'], 'thumbnail', $settings );
			$image_html = '<img src="' . esc_attr( $image_url ) . '" alt="' . esc_attr( \Elementor\Control_Media::get_image_alt( $attachment ) ) . '" />';
		
			$link_tag = '';

			$link = $this->get_link_url( $attachment, $settings );
			if ( $link ) {
				$link_key = 'link_' . $index;

				$this->add_render_attribute( $link_key, [
					'href' => $link['url'],
					'data-elementor-open-lightbox' => $settings['open_lightbox'],
					'data-elementor-lightbox-slideshow' => $this->get_id(),
					'data-elementor-lightbox-index' => $index,
					'data-effect' => 'fadeInTop',
				] );

				if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
					$this->add_render_attribute( $link_key, [
						'class' => 'elementor-clickable',
					] );
				}

				$this->add_render_attribute( $link_key, [
					'class' => 'has-reveal-effect',
					'style' => 'background-image: url(' . $image_url . ')',
				] );
				$link_tag = '<a ' . $this->get_render_attribute_string( $link_key ) . '>';
			} else {
				$link_key = 'link_' . $index;
				$this->add_render_attribute( $link_key, [
					'class' => 'metro-no-link',
					'style' => 'background-image: url(' . $image_url . ')',
					'data-effect' => 'fadeInTop',
				] );
				$this->add_render_attribute( $link_key, [
					'class' => 'has-reveal-effect',
				] );

				$link_tag = '<span ' . $this->get_render_attribute_string( $link_key ) . '>';
			}

			$this->add_render_attribute( 'img_wrap'.$index, 'class', 'metro-grid-item' );

			if ( $c == 1 && $total > 3 ) {
				$this->add_render_attribute( 'img_wrap'.$index, 'class', 'metro-double' );
			} else if ( $c % 4 == 0 ) {
				$this->add_render_attribute( 'img_wrap'.$index, 'class', 'metro-landscape' );
			} else if ( $c % 5 == 0 ) {
				$this->add_render_attribute( 'img_wrap'.$index, 'class', 'metro-potrait' );
			}

			$slide_html = '<div ' . $this->get_render_attribute_string( 'img_wrap'.$index ) . '>';

			$slide_html .= $link_tag . $image_html;

			if ( $link ) {
				$slide_html .= '</a>';
			} else {
				$slide_html .= '</span>';
			}

			$slide_html .= '</div>';

			$galleries[] = $slide_html;
		}

		if ( empty( $galleries ) ) {
			return;
		}

		?>
		<div class="br-metro-gallery br-gallery">

			<div class="metro-grid">

				<div class="metro__sizer"></div>

				<?php echo implode( "\n\n", $galleries ); ?>

			</div>

		</div>
		<?php
	}

	/**
	 * Retrieve image carousel link URL.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param array $attachment
	 * @param object $instance
	 *
	 * @return array|string|false An array/string containing the attachment URL, or false if no link.
	 */
	private function get_link_url( $attachment, $instance ) {
		if ( 'none' === $instance['gallery_link'] ) {
			return false;
		}

		return [
			'url' => wp_get_attachment_url( $attachment['id'] ),
		];
	}
}
