<?php
/**
 * custom elementor portfolio slide widget
 *
 * @package Elementor
 * @subpackage Brayn Core
 * @since 1.0
 */

/**
 * Elementor portfolio carousel Widget.
 *
 * @since 1.0.0
 */
class Brayn_Elementor_Portfolio_Slide_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-portfolio-slide';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Portfolio Slide', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-slider-album';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_responsive_control(
			'slide_col',
			[
				'label' => esc_html__( 'Column', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'col' => [
						'min' => 1,
						'max' => 9,
						'step' => 1,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 3,
					'unit' => 'col',
				],
				'tablet_default' => [
					'size' => 2,
					'unit' => 'col',
				],
				'mobile_default' => [
					'size' => 1,
					'unit' => 'col',
				],
			]
		);

		$this->add_control(
			'number_post',
			[
				'label' => esc_html__( 'Number of post', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'input_type' => 'number',
				'placeholder' => '',
				'default' => 10,
				'description' => esc_html__( 'Enter "-1" to show all posts', 'brayn-core' ),
			]
		);

		$this->add_control(
			'order_by',
			[
				'label' => esc_html__( 'Order by', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'date',
				'description' => '',
				'options' => [
					'ID'    => esc_html__( 'Post ID', 'brayn-core' ),
					'date'  => esc_html__( 'Post date', 'brayn-core' ),
					'title' => esc_html__( 'Post title', 'brayn-core' ),
					'rand'  => esc_html__( 'Random order', 'brayn-core' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label' => esc_html__( 'Order', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'DESC',
				'description' => '',
				'options' => [
					'ASC'  => esc_html__( 'Ascending', 'brayn-core' ),
					'DESC' => esc_html__( 'Descending', 'brayn-core' ),
				],
			]
		);

		$portfolio__cats = brayn_core_taxonomies_options( 'portfolio-cat' );
		$this->add_control(
			'category',
			[
				'label' => esc_html__( 'Portfolio category', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'all',
				'description' => '',
				'options' => $portfolio__cats,
			]
		);

		$this->add_control(
			'show_portfo_category',
			[
				'label' => esc_html__( 'Display category?', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'On', 'brayn-core' ),
				'label_off' => esc_html__( 'Off', 'brayn-core' ),
				'return_value' => 'yes',
				'default' => '',
				'description' => esc_html__( 'Show/hide portfolio category from list.', 'brayn-core' ),
			]
		);

		$this->add_control(
			'pickup_posts',
			[
				'label' => esc_html__( 'Post ID(s)', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'rows' => 8,
				'placeholder' => '',
				'default' => '',
				'description' => esc_html__( 'Optional: Enter the post ids you want show, one post id per line/row. The "Number of post" & "Portfolio category" value will be ignored if this setting is used.', 'brayn-core' ),
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings    = $this->get_settings_for_display();
		$post_number = $settings['number_post'];
		$order_by    = $settings['order_by'];
		$order       = $settings['order'];
		$category    = $settings['category'];
		$raw_postids = $settings['pickup_posts'];
		$show_porto_cat = $settings['show_portfo_category'];

		$this->add_render_attribute(
			'swipe_wrap',
			[
				'class' => [ 'portfolio-carousel-contain', 'swiper-container' ],
				'data-colnumb' => '{"def":"'.$settings['slide_col']['size'].'","tablet":"'.$settings['slide_col_tablet']['size'].'","mobile":"'.$settings['slide_col_mobile']['size'].'"}',
			]
		);

		$post_args = array(
			'post_type'           => 'portfolio',
			'post_status'         => 'publish',
			'ignore_sticky_posts' => 1,
			'orderby'             => $order_by,
			'order'               => $order,
			'paged'               => 1,
		);

		if ( "" === $raw_postids ) {
			$post_args['posts_per_page'] = intval($post_number);

			if ( $category !== 'all' && intval( $category ) > 0 ) {
				$post_args['tax_query'] = array();
				$post_args['tax_query'][] = array(
						'taxonomy' => 'portfolio-cat',
						'field'    => 'id',
						'terms'    => intval( $category ),
						'operator' => 'IN'
				);
			}
		} else {
			$post_ids = explode( "\n", str_replace("\r", "", $raw_postids) );
			if ( ! empty( $post_ids ) ) {
				$post_args['post__in'] = $post_ids;
			}
		}

		$p_query = new WP_Query( $post_args );
		if ( $p_query->have_posts() ) :

			print '<div ' . $this->get_render_attribute_string( 'swipe_wrap' ) . '>' . "\n";
			?>
			<!-- previous button -->
			<div class="portfolio-carousel-btn-prev">
				<svg class="svg-icon" viewBox="0 0 20 20">
					<path d="M18.271,9.212H3.615l4.184-4.184c0.306-0.306,0.306-0.801,0-1.107c-0.306-0.306-0.801-0.306-1.107,0
					L1.21,9.403C1.194,9.417,1.174,9.421,1.158,9.437c-0.181,0.181-0.242,0.425-0.209,0.66c0.005,0.038,0.012,0.071,0.022,0.109
					c0.028,0.098,0.075,0.188,0.142,0.271c0.021,0.026,0.021,0.061,0.045,0.085c0.015,0.016,0.034,0.02,0.05,0.033l5.484,5.483
					c0.306,0.307,0.801,0.307,1.107,0c0.306-0.305,0.306-0.801,0-1.105l-4.184-4.185h14.656c0.436,0,0.788-0.353,0.788-0.788
					S18.707,9.212,18.271,9.212z"></path>
				</svg>
			</div>

			<!-- next button -->
			<div class="portfolio-carousel-btn-next">
				<svg class="svg-icon" viewBox="0 0 20 20">
					<path d="M1.729,9.212h14.656l-4.184-4.184c-0.307-0.306-0.307-0.801,0-1.107c0.305-0.306,0.801-0.306,1.106,0
					l5.481,5.482c0.018,0.014,0.037,0.019,0.053,0.034c0.181,0.181,0.242,0.425,0.209,0.66c-0.004,0.038-0.012,0.071-0.021,0.109
					c-0.028,0.098-0.075,0.188-0.143,0.271c-0.021,0.026-0.021,0.061-0.045,0.085c-0.015,0.016-0.034,0.02-0.051,0.033l-5.483,5.483
					c-0.306,0.307-0.802,0.307-1.106,0c-0.307-0.305-0.307-0.801,0-1.105l4.184-4.185H1.729c-0.436,0-0.788-0.353-0.788-0.788
					S1.293,9.212,1.729,9.212z"></path>
				</svg>
			</div>

			<div class="swiper-wrapper">

			<?php
			$numb = 0;
			while ( $p_query->have_posts() ) :
				$p_query->the_post();
				global $post;

				$cat_html = false;
				$categories = get_the_term_list( get_the_ID(), 'portfolio-cat', '', ', ' );
				if ( $categories ) {
					$cat_html = strip_tags( $categories );
				}

				$numb++;
					?>
							<div class="swiper-slide">
								<article class="portfolio-carousel-item">
									<div class="portfolio-carousel-obj">
										<a href="<?php the_permalink(); ?>" rel="bookmark">
											<div class="portfolio-carousel-img">
												<?php
													if ( $thumbnail = get_the_post_thumbnail_url( $post, 'brayn-post-slide' ) ) {
														print '<img src="' . $thumbnail . '" alt=""/>';
													} else {
														print '<img src="' . esc_url( BRAYN_CORE_URL . '/assets/images/image.jpg' ) . '" alt=""/>';
													}
												?>
											</div>
											<header>
												<h3><span class="numbering"><?php echo sprintf('%02d', $numb); ?></span> <?php the_title(); ?></h3>
												<?php if ( $show_porto_cat && $cat_html ) { ?><span class="portfolio-category"><?php print esc_html( $cat_html ); ?></span><?php } ?>
											</header>
										</a>
									</div>
								</article>
							</div>
					<?php
			endwhile;
			wp_reset_postdata();
			$numb = 0;
			unset( $numb );

			print '</div>' . "\n";

			print '</div>' . "\n";

		endif;
	}
}
