<?php
/**
 * custom elementor portfolio widget
 *
 * @package    Elementor
 * @subpackage Brayn Core
 * @since      1.0
 */

/**
 * Elementor portfolio Widget.
 *
 * Elementor widget that inserts portfolio post type.
 *
 * @since 1.0.0
 */
class Brayn_Elementor_Portfolio_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve portfolio widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		$name = 'brayn-portfolio';
		return $name;
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve oEmbed widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Portfolio', 'brayn-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve oEmbed widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-posts-group';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the oEmbed widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'brayn-theme' ];
	}

	/**
	 * Register oEmbed widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Settings', 'brayn-core' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'number_post',
			[
				'label' => esc_html__( 'Number of post', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'input_type' => 'number',
				'placeholder' => '',
				'default' => 10,
				'description' => esc_html__( 'Enter "-1" to show all posts', 'brayn-core' ),
			]
		);

		$this->add_control(
			'order_by',
			[
				'label' => esc_html__( 'Order by', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'date',
				'description' => '',
				'options' => [
					'ID'    => esc_html__( 'Post ID', 'brayn-core' ),
					'date'  => esc_html__( 'Post date', 'brayn-core' ),
					'title' => esc_html__( 'Post title', 'brayn-core' ),
					'rand'  => esc_html__( 'Random order', 'brayn-core' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label' => esc_html__( 'Order', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'DESC',
				'description' => '',
				'options' => [
					'ASC'  => esc_html__( 'Ascending', 'brayn-core' ),
					'DESC' => esc_html__( 'Descending', 'brayn-core' ),
				],
			]
		);

		$portfolio__cats = brayn_core_taxonomies_options( 'portfolio-cat' );
		$this->add_control(
			'category',
			[
				'label' => esc_html__( 'Portfolio category', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'all',
				'description' => '',
				'options' => $portfolio__cats,
			]
		);

		$this->add_control(
			'show_portfolio_category',
			[
				'label' => esc_html__( 'Display category?', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'On', 'brayn-core' ),
				'label_off' => esc_html__( 'Off', 'brayn-core' ),
				'return_value' => 'yes',
				'default' => '',
				'description' => esc_html__( 'Show/hide portfolio category from list.', 'brayn-core' ),
			]
		);

		$this->add_control(
			'show_pagination',
			[
				'label' => esc_html__( 'Pagination', 'brayn-core' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'On', 'brayn-core' ),
				'label_off' => esc_html__( 'Off', 'brayn-core' ),
				'return_value' => 'yes',
				'default' => '',
				'description' => esc_html__( 'Enable/disable pagination. Please prevent to enable multiple paginations in the same page.', 'brayn-core' ),
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render oEmbed widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings    = $this->get_settings_for_display();
		$post_number = $settings['number_post'];
		$order_by    = $settings['order_by'];
		$order       = $settings['order'];
		$category    = $settings['category'];
		$pagination  = $settings['show_pagination'];
		$show_cat    = $settings['show_portfolio_category'];

		$post_args = array(
			'post_type'           => 'portfolio',
			'post_status'         => 'publish',
			'posts_per_page'      => intval($post_number),
			'ignore_sticky_posts' => 1,
			'orderby'             => $order_by,
			'order'               => $order,
		);

		if ( 'yes' === $pagination ) {
			$paged = ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1;
			if ( is_front_page() ) {
				$paged = ( get_query_var( 'page' ) ) ? absint( get_query_var( 'page' ) ) : 1;
			}
			if ( isset( $_GET['custom_paged'] ) ) {
				$paged = (int) $_GET['custom_paged'];
			}
			$post_args['paged'] = $paged;
		} else {
			// no pagination? we should stick to page 1 then.
			$post_args['paged'] = 1;
		}

		if ( $category !== 'all' && intval( $category ) > 0 ) {
			$post_args['tax_query'] = array();
			$post_args['tax_query'][] = array(
					'taxonomy' => 'portfolio-cat',
					'field'    => 'id',
					'terms'    => intval( $category ),
					'operator' => 'IN'
			);
		}

		$p_query = new WP_Query( $post_args );
		if ( $p_query->have_posts() ) :

			print '<div class="portfolio-grid-wrapper">' . "\n";
			print '<div class="portfolio-grid-lists">' . "\n";

			$GLOBALS['show_cat'] = $show_cat;
			while ( $p_query->have_posts() ) :
				$p_query->the_post();

				get_template_part( 'template-parts/portfolio/loop', '' );

			endwhile;

			if ( isset( $GLOBALS['show_cat'] ) ) {
				unset( $GLOBALS['show_cat'] );
			}

			print '</div>' . "\n";

			if ( 'yes' === $pagination ) {
				the_brayn_core_custom_paginate_links( array(
					'prev_text' => '<svg style="enable-background:new 0 0 50 50;" viewBox="0 0 50 50"><g><polygon points="37.561,47.293 15.267,25 37.561,2.707 36.146,1.293 12.439,25 36.146,48.707"/></g></svg><span class="screen-reader-text">' . esc_html__( 'Previous page', 'brayn-core' ) . '</span>',
					'next_text' => '<span class="screen-reader-text">' . esc_html__( 'Next page', 'brayn-core' ) . '</span><svg style="enable-background:new 0 0 50 50;" viewBox="0 0 50 50"><g><polygon points="13.854,48.707 37.561,25 13.854,1.293 12.439,2.707 34.732,25 12.439,47.293"/></g></svg>',
					'before_page_number' => '<span class="meta-nav screen-reader-text">' . esc_html__( 'Page', 'brayn-core' ) . ' </span>',
					'current' => ( is_front_page() ) ? max( 1, get_query_var('page') ) : max( 1, get_query_var('paged') ),
					'total' => $p_query->max_num_pages,
				) );
			}

			wp_reset_postdata();

			print '</div>' . "\n";
		else :

			print '<h4>'. esc_html__( 'No posts found!', 'brayn-core' ).'</h4>';

		endif;
	}
}
